import { isWebType } from './loader.js';
import { ensurePathStartsWithSlash } from './validation/common.js';
import { isType } from '../../utilities/types.js';
import { zod } from '@shopify/cli-kit/node/schema';
import { getDependencies, readAndParsePackageJson } from '@shopify/cli-kit/node/node-package-manager';
import { fileRealPath, findPathUp } from '@shopify/cli-kit/node/fs';
import { joinPath } from '@shopify/cli-kit/node/path';
import { AbortError } from '@shopify/cli-kit/node/error';
import { getPathValue } from '@shopify/cli-kit/common/object';
export const LegacyAppSchema = zod
    .object({
    client_id: zod.number().optional(),
    name: zod.string().optional(),
    scopes: zod.string().default(''),
    extension_directories: zod.array(zod.string()).optional(),
    web_directories: zod.array(zod.string()).optional(),
})
    .strict();
export const AppSchema = zod.object({
    client_id: zod.string(),
    build: zod
        .object({
        automatically_update_urls_on_dev: zod.boolean().optional(),
        dev_store_url: zod.string().optional(),
        include_config_on_deploy: zod.boolean().optional(),
    })
        .optional(),
    extension_directories: zod.array(zod.string()).optional(),
    web_directories: zod.array(zod.string()).optional(),
});
export const AppConfigurationSchema = zod.union([LegacyAppSchema, AppSchema]);
export function getAppVersionedSchema(specs) {
    const isConfigSpecification = (spec) => spec.experience === 'configuration';
    const schema = specs
        .filter(isConfigSpecification)
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        .reduce((schema, spec) => schema.merge(spec.schema), AppSchema);
    return specs.length > 0 ? schema.strict() : schema;
}
/**
 * Check whether a shopify.app.toml schema is valid against the legacy schema definition.
 * @param item - the item to validate
 */
export function isLegacyAppSchema(item) {
    const { path, ...rest } = item;
    return isType(LegacyAppSchema, rest);
}
/**
 * Check whether a shopify.app.toml schema is valid against the current schema definition.
 * @param item - the item to validate
 */
export function isCurrentAppSchema(item) {
    const { path, ...rest } = item;
    return isType(AppSchema.nonstrict(), rest);
}
/**
 * Get scopes from a given app.toml config file.
 * @param config - a configuration file
 */
export function getAppScopes(config) {
    if (isLegacyAppSchema(config)) {
        return config.scopes;
    }
    else if (isCurrentAppSchema(config)) {
        return config.access_scopes?.scopes ?? '';
    }
    return '';
}
/**
 * Get scopes as an array from a given app.toml config file.
 * @param config - a configuration file
 */
export function getAppScopesArray(config) {
    const scopes = getAppScopes(config);
    return scopes.length ? scopes.split(',').map((scope) => scope.trim()) : [];
}
export function usesLegacyScopesBehavior(config) {
    if (isLegacyAppSchema(config))
        return true;
    if (isCurrentAppSchema(config))
        return config.access_scopes?.use_legacy_install_flow ?? false;
    return false;
}
export function appIsLaunchable(app) {
    const frontendConfig = app?.webs?.find((web) => isWebType(web, WebType.Frontend));
    const backendConfig = app?.webs?.find((web) => isWebType(web, WebType.Backend));
    return Boolean(frontendConfig || backendConfig);
}
export function filterNonVersionedAppFields(configuration) {
    return Object.keys(configuration).filter((fieldName) => !Object.keys(AppSchema.shape).concat('path').includes(fieldName));
}
export var WebType;
(function (WebType) {
    WebType["Frontend"] = "frontend";
    WebType["Backend"] = "backend";
    WebType["Background"] = "background";
})(WebType || (WebType = {}));
const WebConfigurationAuthCallbackPathSchema = zod.preprocess(ensurePathStartsWithSlash, zod.string());
const baseWebConfigurationSchema = zod.object({
    auth_callback_path: zod
        .union([WebConfigurationAuthCallbackPathSchema, WebConfigurationAuthCallbackPathSchema.array()])
        .optional(),
    webhooks_path: zod.preprocess(ensurePathStartsWithSlash, zod.string()).optional(),
    port: zod.number().max(65536).min(0).optional(),
    commands: zod.object({
        build: zod.string().optional(),
        dev: zod.string(),
    }),
    name: zod.string().optional(),
    hmr_server: zod.object({ http_paths: zod.string().array() }).optional(),
});
const webTypes = zod.enum([WebType.Frontend, WebType.Backend, WebType.Background]).default(WebType.Frontend);
export const WebConfigurationSchema = zod.union([
    baseWebConfigurationSchema.extend({ roles: zod.array(webTypes) }),
    baseWebConfigurationSchema.extend({ type: webTypes }),
]);
export const ProcessedWebConfigurationSchema = baseWebConfigurationSchema.extend({ roles: zod.array(webTypes) });
export class App {
    constructor({ name, idEnvironmentVariableName, directory, packageManager, configuration, nodeDependencies, webs, modules, usesWorkspaces, dotenv, errors, specifications, configSchema, remoteBetaFlags, }) {
        this.name = name;
        this.idEnvironmentVariableName = idEnvironmentVariableName;
        this.directory = directory;
        this.packageManager = packageManager;
        this.configuration = this.configurationTyped(configuration);
        this.nodeDependencies = nodeDependencies;
        this.webs = webs;
        this.dotenv = dotenv;
        this.realExtensions = modules;
        this.errors = errors;
        this.usesWorkspaces = usesWorkspaces;
        this.specifications = specifications;
        this.configSchema = configSchema ?? AppSchema;
        this.remoteBetaFlags = remoteBetaFlags ?? [];
    }
    get allExtensions() {
        return this.realExtensions.filter((ext) => !ext.isAppConfigExtension || this.includeConfigOnDeploy);
    }
    get draftableExtensions() {
        return this.realExtensions.filter((ext) => ext.isDraftable());
    }
    async updateDependencies() {
        const nodeDependencies = await getDependencies(joinPath(this.directory, 'package.json'));
        this.nodeDependencies = nodeDependencies;
    }
    async preDeployValidation() {
        const functionExtensionsWithUiHandle = this.allExtensions.filter((ext) => ext.isFunctionExtension && ext.configuration.ui?.handle);
        if (functionExtensionsWithUiHandle.length > 0) {
            const errors = validateFunctionExtensionsWithUiHandle(functionExtensionsWithUiHandle, this.allExtensions);
            if (errors) {
                throw new AbortError('Invalid function configuration', errors.join('\n'));
            }
        }
        await Promise.all([this.allExtensions.map((ext) => ext.preDeployValidation())]);
    }
    hasExtensions() {
        return this.allExtensions.length > 0;
    }
    extensionsForType(specification) {
        return this.allExtensions.filter((extension) => extension.type === specification.identifier || extension.type === specification.externalIdentifier);
    }
    updateExtensionUUIDS(uuids) {
        this.allExtensions.forEach((extension) => {
            extension.devUUID = uuids[extension.localIdentifier] ?? extension.devUUID;
        });
    }
    get includeConfigOnDeploy() {
        if (isLegacyAppSchema(this.configuration))
            return false;
        return this.configuration.build?.include_config_on_deploy;
    }
    configurationTyped(configuration) {
        if (isLegacyAppSchema(configuration))
            return configuration;
        return {
            ...configuration,
            ...buildSpecsAppConfiguration(configuration),
        };
    }
}
export function buildSpecsAppConfiguration(content) {
    return {
        ...homeConfiguration(content),
        ...appProxyConfiguration(content),
        ...posConfiguration(content),
        ...webhooksConfiguration(content),
        ...accessConfiguration(content),
    };
}
function appProxyConfiguration(configuration) {
    if (!getPathValue(configuration, 'app_proxy'))
        return;
    return {
        app_proxy: {
            url: getPathValue(configuration, 'app_proxy.url'),
            prefix: getPathValue(configuration, 'app_proxy.prefix'),
            subpath: getPathValue(configuration, 'app_proxy.subpath'),
        },
    };
}
function homeConfiguration(configuration) {
    const appPreferencesUrl = getPathValue(configuration, 'app_preferences.url');
    return {
        name: getPathValue(configuration, 'name'),
        application_url: getPathValue(configuration, 'application_url'),
        embedded: getPathValue(configuration, 'embedded'),
        ...(appPreferencesUrl ? { app_preferences: { url: appPreferencesUrl } } : {}),
    };
}
function posConfiguration(configuration) {
    const embedded = getPathValue(configuration, 'pos.embedded');
    return embedded === undefined
        ? undefined
        : {
            pos: {
                embedded,
            },
        };
}
function webhooksConfiguration(configuration) {
    return {
        webhooks: { ...getPathValue(configuration, 'webhooks') },
    };
}
function accessConfiguration(configuration) {
    const scopes = getPathValue(configuration, 'access_scopes.scopes');
    const useLegacyInstallFlow = getPathValue(configuration, 'access_scopes.use_legacy_install_flow');
    const redirectUrls = getPathValue(configuration, 'auth.redirect_urls');
    return {
        ...(scopes || useLegacyInstallFlow ? { access_scopes: { scopes, use_legacy_install_flow: useLegacyInstallFlow } } : {}),
        ...(redirectUrls ? { auth: { redirect_urls: redirectUrls } } : {}),
    };
}
export function validateFunctionExtensionsWithUiHandle(functionExtensionsWithUiHandle, allExtensions) {
    const errors = [];
    functionExtensionsWithUiHandle.forEach((extension) => {
        const uiHandle = extension.configuration.ui.handle;
        const matchingExtension = findExtensionByHandle(allExtensions, uiHandle);
        if (!matchingExtension) {
            errors.push(`[${extension.name}] - Local app must contain a ui_extension with handle '${uiHandle}'`);
        }
        else if (matchingExtension.configuration.type !== 'ui_extension') {
            errors.push(`[${extension.name}] - Local app must contain one extension of type 'ui_extension' and handle '${uiHandle}'`);
        }
    });
    return errors.length > 0 ? errors : undefined;
}
function findExtensionByHandle(allExtensions, handle) {
    return allExtensions.find((ext) => ext.handle === handle);
}
export class EmptyApp extends App {
    constructor(specifications, betas, clientId) {
        const configuration = clientId
            ? { client_id: clientId, access_scopes: { scopes: '' }, path: '' }
            : { scopes: '', path: '' };
        const configSchema = getAppVersionedSchema(specifications ?? []);
        super({
            name: '',
            idEnvironmentVariableName: '',
            directory: '',
            packageManager: 'npm',
            configuration,
            nodeDependencies: {},
            webs: [],
            modules: [],
            usesWorkspaces: false,
            specifications,
            configSchema,
            remoteBetaFlags: betas ?? [],
        });
    }
}
/**
 * Given a UI extension, it returns the version of the renderer package.
 * Looks for `/node_modules/@shopify/{renderer-package-name}/package.json` to find the real version used.
 * @param extension - UI extension whose renderer version will be obtained.
 * @returns The version if the dependency exists.
 */
export async function getUIExtensionRendererVersion(extension) {
    // Look for the vanilla JS version of the dependency (the react one depends on it, will always be present)
    const rendererDependency = extension.dependency;
    if (!rendererDependency)
        return undefined;
    return getDependencyVersion(rendererDependency, extension.directory);
}
export async function getDependencyVersion(dependency, directory) {
    // Split the dependency name to avoid using "/" in windows. Only look for non react dependencies.
    const dependencyName = dependency.replace('-react', '').split('/');
    const pattern = joinPath('node_modules', dependencyName[0], dependencyName[1], 'package.json');
    let packagePath = await findPathUp(pattern, {
        cwd: directory,
        type: 'file',
        allowSymlinks: true,
    });
    if (!packagePath)
        return 'not_found';
    packagePath = await fileRealPath(packagePath);
    // Load the package.json and extract the version
    const packageContent = await readAndParsePackageJson(packagePath);
    if (!packageContent.version)
        return 'not_found';
    return { name: dependency, version: packageContent.version };
}
//# sourceMappingURL=app.js.map