import { App, WebType, getAppVersionedSchema } from './app.js';
import themeExtension from '../templates/theme-specifications/theme.js';
import { ExtensionInstance } from '../extensions/extension-instance.js';
import { loadLocalExtensionsSpecifications } from '../extensions/load-specifications.js';
import productSubscriptionUIExtension from '../templates/ui-specifications/product_subscription.js';
import webPixelUIExtension from '../templates/ui-specifications/web_pixel_extension.js';
export const DEFAULT_CONFIG = {
    path: '/tmp/project/shopify.app.toml',
    application_url: 'https://myapp.com',
    client_id: '12345',
    name: 'my app',
    webhooks: {
        api_version: '2023-04',
    },
    embedded: true,
    access_scopes: {
        scopes: 'read_products',
    },
};
export function testApp(app = {}, schemaType = 'legacy') {
    const getConfig = () => {
        if (schemaType === 'legacy') {
            return { scopes: '', extension_directories: [], path: '' };
        }
        else {
            return DEFAULT_CONFIG;
        }
    };
    const newApp = new App({
        name: app.name ?? 'App',
        idEnvironmentVariableName: app.idEnvironmentVariableName ?? 'SHOPIFY_API_KEY',
        directory: app.directory ?? '/tmp/project',
        packageManager: app.packageManager ?? 'yarn',
        configuration: app.configuration ?? getConfig(),
        nodeDependencies: app.nodeDependencies ?? {},
        webs: app.webs ?? [
            {
                directory: '',
                configuration: {
                    roles: [WebType.Backend],
                    commands: { dev: '' },
                },
            },
        ],
        modules: app.allExtensions ?? [],
        usesWorkspaces: app.usesWorkspaces ?? false,
        dotenv: app.dotenv,
        errors: app.errors,
        specifications: app.specifications,
        configSchema: app.configSchema,
    });
    if (app.updateDependencies) {
        Object.getPrototypeOf(newApp).updateDependencies = app.updateDependencies;
    }
    if (app.extensionsForType) {
        Object.getPrototypeOf(newApp).extensionsForType = app.extensionsForType;
    }
    return newApp;
}
export function testAppWithLegacyConfig({ app = {}, config = {} }) {
    const configuration = {
        path: '',
        scopes: '',
        name: 'name',
        extension_directories: [],
        ...config,
    };
    return testApp({ ...app, configuration });
}
export function testAppWithConfig(options) {
    const app = testApp(options?.app, 'current');
    app.configuration = {
        ...DEFAULT_CONFIG,
        ...options?.config,
    };
    return app;
}
export function getWebhookConfig(webhookConfigOverrides) {
    return {
        ...DEFAULT_CONFIG,
        webhooks: {
            ...DEFAULT_CONFIG.webhooks,
            ...webhookConfigOverrides,
        },
    };
}
export function testOrganizationApp(app = {}) {
    const defaultApp = {
        id: '1',
        title: 'app1',
        apiKey: 'api-key',
        apiSecretKeys: [{ secret: 'api-secret' }],
        organizationId: '1',
        grantedScopes: [],
        applicationUrl: 'https://example.com',
        redirectUrlWhitelist: ['https://example.com/callback1'],
        disabledBetas: [],
        betas: [],
    };
    return { ...defaultApp, ...app };
}
export async function testUIExtension(uiExtension = {}) {
    const directory = uiExtension?.directory ?? '/tmp/project/extensions/test-ui-extension';
    const configuration = uiExtension?.configuration ?? {
        name: uiExtension?.configuration?.name ?? 'test-ui-extension',
        type: uiExtension?.configuration?.type ?? uiExtension?.type ?? 'product_subscription',
        metafields: [],
        capabilities: {
            block_progress: false,
            network_access: false,
            api_access: false,
            collect_buyer_consent: {
                sms_marketing: false,
                customer_privacy: false,
            },
        },
        targeting: [{ target: 'target1' }, { target: 'target2' }],
    };
    const configurationPath = uiExtension?.configurationPath ?? `${directory}/shopify.ui.extension.toml`;
    const entryPath = uiExtension?.entrySourceFilePath ?? `${directory}/src/index.js`;
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === configuration.type);
    const extension = new ExtensionInstance({
        configuration: configuration,
        configurationPath,
        entryPath,
        directory,
        specification,
    });
    extension.devUUID = uiExtension?.devUUID ?? 'test-ui-extension-uuid';
    return extension;
}
export async function testThemeExtensions(directory = './my-extension') {
    const configuration = {
        name: 'theme extension name',
        type: 'theme',
        metafields: [],
    };
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === 'theme');
    const extension = new ExtensionInstance({
        configuration,
        configurationPath: '',
        directory,
        specification,
    });
    return extension;
}
export async function testAppConfigExtensions(emptyConfig = false) {
    const configuration = emptyConfig
        ? {}
        : {
            pos: {
                embedded: true,
            },
        };
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === 'point_of_sale');
    const extension = new ExtensionInstance({
        configuration,
        configurationPath: 'shopify.app.toml',
        directory: './',
        specification,
    });
    return extension;
}
export async function testPaymentExtensions(directory = './my-extension') {
    const configuration = {
        name: 'Payment Extension Name',
        type: 'payments_extension',
        targeting: [{ target: 'payments.offsite.render' }],
        metafields: [],
    };
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === 'payments_extension');
    const extension = new ExtensionInstance({
        configuration,
        configurationPath: '',
        directory,
        specification,
    });
    return extension;
}
export async function testWebhookExtensions(emptyConfig = false) {
    const configuration = emptyConfig
        ? {}
        : {
            webhooks: {
                subscriptions: [
                    {
                        topics: ['orders/delete'],
                        uri: 'https://my-app.com/webhooks',
                    },
                ],
            },
        };
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === 'webhooks');
    const extension = new ExtensionInstance({
        configuration,
        configurationPath: '',
        directory: './',
        specification,
    });
    return extension;
}
export async function testWebPixelExtension(directory = './my-extension') {
    const configuration = {
        name: 'web pixel name',
        type: 'web_pixel',
        metafields: [],
        runtime_context: 'strict',
        customer_privacy: {
            analytics: false,
            marketing: true,
            preferences: false,
            sale_of_data: 'enabled',
        },
        settings: [],
    };
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === 'web_pixel_extension');
    const parsed = specification.schema.parse(configuration);
    const extension = new ExtensionInstance({
        configuration: parsed,
        configurationPath: '',
        directory,
        specification,
    });
    return extension;
}
export async function testTaxCalculationExtension(directory = './my-extension') {
    const configuration = {
        name: 'tax',
        type: 'tax_calculation',
        metafields: [],
        runtime_context: 'strict',
        customer_privacy: {
            analytics: false,
            marketing: true,
            preferences: false,
            sale_of_data: 'enabled',
        },
    };
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === 'tax_calculation');
    const extension = new ExtensionInstance({
        configuration,
        configurationPath: '',
        directory,
        specification,
    });
    return extension;
}
export async function testFlowActionExtension(directory = './my-extension') {
    const configuration = {
        name: 'flow action',
        type: 'flow_action',
        metafields: [],
        runtime_context: 'strict',
    };
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === 'flow_action');
    const extension = new ExtensionInstance({
        configuration,
        configurationPath: '',
        directory,
        specification,
    });
    return extension;
}
function defaultFunctionConfiguration() {
    return {
        name: 'test function extension',
        description: 'description',
        type: 'product_discounts',
        build: {
            command: 'echo "hello world"',
        },
        api_version: '2022-07',
        configuration_ui: true,
        metafields: [],
    };
}
export async function testFunctionExtension(opts = {}) {
    const directory = opts.dir ?? '/tmp/project/extensions/my-function';
    const configuration = opts.config ?? defaultFunctionConfiguration();
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === 'function');
    const extension = new ExtensionInstance({
        configuration,
        configurationPath: '',
        entryPath: opts.entryPath,
        directory,
        specification,
    });
    return extension;
}
export async function testPaymentsAppExtension(opts) {
    const directory = opts.dir ?? '/tmp/project/extensions/my-payments-app-extension';
    const configuration = opts.config;
    const allSpecs = await loadLocalExtensionsSpecifications();
    const specification = allSpecs.find((spec) => spec.identifier === 'payments_extension');
    const extension = new ExtensionInstance({
        configuration,
        configurationPath: '',
        entryPath: opts.entryPath,
        directory,
        specification,
    });
    return extension;
}
export const testRemoteSpecifications = [
    {
        name: 'Checkout Post Purchase',
        externalName: 'Post-purchase UI',
        identifier: 'checkout_post_purchase',
        externalIdentifier: 'checkout_post_purchase_external',
        gated: false,
        experience: 'extension',
        options: {
            managementExperience: 'cli',
            registrationLimit: 1,
        },
        features: {
            argo: {
                surface: 'checkout',
            },
        },
    },
    {
        name: 'Online Store - App Theme Extension',
        externalName: 'Theme App Extension',
        identifier: 'theme',
        externalIdentifier: 'theme_external',
        gated: false,
        experience: 'extension',
        options: {
            managementExperience: 'cli',
            registrationLimit: 1,
        },
    },
    {
        name: 'Product Subscription',
        externalName: 'Subscription UI',
        identifier: 'product_subscription',
        externalIdentifier: 'product_subscription_external',
        gated: false,
        experience: 'extension',
        options: {
            managementExperience: 'cli',
            registrationLimit: 1,
        },
        features: {
            argo: {
                surface: 'admin',
            },
        },
    },
    {
        name: 'UI Extension',
        externalName: 'UI Extension',
        identifier: 'ui_extension',
        externalIdentifier: 'ui_extension_external',
        gated: false,
        experience: 'extension',
        options: {
            managementExperience: 'cli',
            registrationLimit: 50,
        },
        features: {
            argo: {
                surface: 'all',
            },
        },
    },
    {
        name: 'Checkout Extension',
        externalName: 'Checkout UI',
        identifier: 'checkout_ui_extension',
        externalIdentifier: 'checkout_ui_extension_external',
        gated: false,
        experience: 'extension',
        options: {
            managementExperience: 'cli',
            registrationLimit: 5,
        },
        features: {
            argo: {
                surface: 'checkout',
            },
        },
    },
    {
        name: 'Product Subscription',
        externalName: 'Subscription UI',
        // we are going to replace this to 'product_subscription' because we
        // started using it before relying on the extension specification identifier
        identifier: 'subscription_management',
        externalIdentifier: 'product_subscription_external',
        gated: false,
        experience: 'extension',
        options: {
            managementExperience: 'cli',
            registrationLimit: 1,
        },
        features: {
            argo: {
                surface: 'admin',
            },
        },
    },
    {
        name: 'Marketing Activity',
        externalName: 'Marketing Activity',
        identifier: 'marketing_activity_extension',
        externalIdentifier: 'marketing_activity_extension_external',
        gated: false,
        experience: 'extension',
        options: {
            managementExperience: 'dashboard',
            registrationLimit: 100,
        },
    },
    {
        name: 'function',
        externalName: 'function',
        identifier: 'function',
        externalIdentifier: 'function',
        gated: false,
        experience: 'extension',
        options: {
            managementExperience: 'cli',
            registrationLimit: 1,
        },
        features: {
            argo: {
                surface: 'checkout',
            },
        },
    },
];
export const testRemoteExtensionTemplates = [
    {
        identifier: 'cart_checkout_validation',
        name: 'Function - Cart and Checkout Validation',
        defaultName: 'cart-checkout-validation',
        group: 'Discounts and checkout',
        supportLinks: ['https://shopify.dev/docs/api/functions/reference/cart-checkout-validation'],
        types: [
            {
                type: 'function',
                url: 'https://github.com/Shopify/function-examples',
                extensionPoints: [],
                supportedFlavors: [
                    {
                        name: 'Rust',
                        value: 'rust',
                        path: 'checkout/rust/cart-checkout-validation/default',
                    },
                ],
            },
        ],
    },
    {
        identifier: 'cart_transform',
        name: 'Function - Cart transformer',
        defaultName: 'cart-transformer',
        group: 'Discounts and checkout',
        supportLinks: [],
        types: [
            {
                type: 'function',
                url: 'https://github.com/Shopify/function-examples',
                extensionPoints: [],
                supportedFlavors: [
                    {
                        name: 'Wasm',
                        value: 'wasm',
                        path: 'checkout/wasm/cart-transform/default',
                    },
                    {
                        name: 'Rust',
                        value: 'rust',
                        path: 'checkout/rust/cart-transform/default',
                    },
                ],
            },
        ],
    },
    {
        identifier: 'product_discounts',
        name: 'Function - Product discounts',
        defaultName: 'product-discounts',
        group: 'Discounts and checkout',
        supportLinks: ['https://shopify.dev/docs/apps/discounts'],
        types: [
            {
                type: 'function',
                url: 'https://github.com/Shopify/function-examples',
                extensionPoints: [],
                supportedFlavors: [
                    {
                        name: 'Wasm',
                        value: 'wasm',
                        path: 'discounts/wasm/product-discounts/default',
                    },
                    {
                        name: 'Rust',
                        value: 'rust',
                        path: 'discounts/rust/product-discounts/default',
                    },
                ],
            },
        ],
    },
    {
        identifier: 'order_discounts',
        name: 'Function - Order discounts',
        defaultName: 'order-discounts',
        group: 'Discounts and checkout',
        supportLinks: [],
        types: [
            {
                type: 'function',
                url: 'https://github.com/Shopify/function-examples',
                extensionPoints: [],
                supportedFlavors: [
                    {
                        name: 'Wasm',
                        value: 'wasm',
                        path: 'discounts/wasm/order-discounts/default',
                    },
                    {
                        name: 'Rust',
                        value: 'rust',
                        path: 'discounts/rust/order-discounts/default',
                    },
                    {
                        name: 'JavaScript',
                        value: 'vanilla-js',
                        path: 'discounts/javascript/order-discounts/default',
                    },
                ],
            },
        ],
    },
];
export const testLocalExtensionTemplates = [
    themeExtension,
    productSubscriptionUIExtension,
    webPixelUIExtension,
];
export const testPartnersUserSession = {
    token: 'token',
    accountInfo: {
        type: 'UserAccount',
        email: 'partner@shopify.com',
    },
};
export const testPartnersServiceSession = {
    token: 'partnersToken',
    accountInfo: {
        type: 'ServiceAccount',
        orgName: 'organization',
    },
};
export async function buildVersionedAppSchema() {
    const configSpecifications = await configurationSpecifications();
    return {
        schema: getAppVersionedSchema(configSpecifications),
        configSpecifications,
    };
}
export async function configurationSpecifications() {
    return (await loadLocalExtensionsSpecifications()).filter((spec) => spec.experience === 'configuration');
}
//# sourceMappingURL=app.test-data.js.map