import { BaseSchema } from './schemas.js';
import { blocks } from '../../constants.js';
import { capitalize } from '@shopify/cli-kit/common/string';
import { zod } from '@shopify/cli-kit/node/schema';
import { getPathValue, setPathValue } from '@shopify/cli-kit/common/object';
/**
 * Create a new ui extension spec.
 *
 * Everything but "identifer" is optional.
 * ```ts
 * identifier: string // unique identifier for the extension type
 * externalIdentifier: string // identifier used externally (default: same as "identifier")
 * partnersWebIdentifier: string // identifier used in the partners web UI (default: same as "identifier")
 * surface?: string // surface where the extension is going to be rendered (default: 'unknown')
 * dependency?: {name: string; version: string} // dependency to be added to the extension's package.json
 * graphQLType?: string // GraphQL type of the extension (default: same as "identifier")
 * schema?: ZodSchemaType<TConfiguration> // schema used to validate the extension's configuration (default: BaseUIExtensionSchema)
 * getBundleExtensionStdinContent?: (configuration: TConfiguration) => string // function to generate the content of the stdin file used to bundle the extension
 * validate?: (configuration: TConfiguration, directory: string) => Promise<Result<undefined, Error>> // function to validate the extension's configuration
 * preDeployValidation?: (configuration: TConfiguration) => Promise<void> // function to validate the extension's configuration before deploying it
 * deployConfig?: (configuration: TConfiguration, directory: string) => Promise<{[key: string]: unknown}> // function to generate the extensions configuration payload to be deployed
 * hasExtensionPointTarget?: (configuration: TConfiguration, target: string) => boolean // function to determine if the extension has a given extension point target
 * ```
 */
export function createExtensionSpecification(spec) {
    const defaults = {
        // these two fields are going to be overridden by the extension specification API response,
        // but we need them to have a default value for tests
        externalIdentifier: `${spec.identifier}_external`,
        additionalIdentifiers: [],
        externalName: capitalize(spec.identifier.replace(/_/g, ' ')),
        surface: 'test-surface',
        partnersWebIdentifier: spec.identifier,
        schema: BaseSchema,
        registrationLimit: blocks.extensions.defaultRegistrationLimit,
        transform: spec.transform,
        reverseTransform: spec.reverseTransform,
        experience: spec.experience ?? 'extension',
    };
    return { ...defaults, ...spec };
}
/**
 * Create a new app config extension spec. This factory method for creating app config extensions is created for two
 * reasons:
 *   - schema needs to be casted to ZodSchemaType<TConfiguration>
 *   - App config extensions have default transform and reverseTransform functions

 */
export function createConfigExtensionSpecification(spec) {
    const appModuleFeatures = spec.appModuleFeatures ?? (() => []);
    return createExtensionSpecification({
        identifier: spec.identifier,
        // This casting is required because `name` and `type` are mandatory for the existing extension spec configurations,
        // however, app config extensions config content is parsed from the `shopify.app.toml`
        schema: spec.schema,
        appModuleFeatures,
        transform: resolveAppConfigTransform(spec.transformConfig),
        reverseTransform: resolveReverseAppConfigTransform(spec.schema, spec.transformConfig),
        experience: 'configuration',
    });
}
function resolveAppConfigTransform(transformConfig) {
    if (!transformConfig)
        return (content) => defaultAppConfigTransform(content);
    if (Object.keys(transformConfig).includes('forward')) {
        return transformConfig.forward;
    }
    else {
        return (content) => appConfigTransform(content, transformConfig);
    }
}
function resolveReverseAppConfigTransform(
// eslint-disable-next-line @typescript-eslint/no-explicit-any
schema, transformConfig) {
    if (!transformConfig)
        return (content) => defaultAppConfigReverseTransform(schema, content);
    if (Object.keys(transformConfig).includes('reverse')) {
        return transformConfig.reverse;
    }
    else {
        return (content) => appConfigTransform(content, transformConfig, true);
    }
}
/**
 * Given an object:
 * ```json
 * { source: { fieldSourceA: 'valueA' } }
 * ```
 *  and a transform config content like this:
 * ```json
 * { 'target.fieldTargetA': 'source.fieldSourceA'}
 * ```
 * the method returns the following object:
 * ```json
 * { source: { fieldTargetA: 'valueA' } }
 * ```
 * The transformation can be applied in both ways depending on the reverse parameter
 *
 * @param content - The objet to be transformed
 * @param config - The transformation config
 * @param reverse - If true, the transformation will be applied in reverse
 *
 * @returns the transformed object
 */
function appConfigTransform(content, config, reverse = false) {
    const transformedContent = {};
    for (const [mappedPath, objectPath] of Object.entries(config)) {
        const originPath = reverse ? mappedPath : objectPath;
        const targetPath = reverse ? objectPath : mappedPath;
        const sourceValue = getPathValue(content, originPath);
        if (sourceValue !== undefined)
            setPathValue(transformedContent, targetPath, sourceValue);
    }
    return transformedContent;
}
/**
 * Flat the configuration object to a single level object. This is the schema expected by the server side.
 * ```json
 * {
 *   pos: {
 *    embedded = true
 *   }
 * }
 * ```
 * will be flattened to:
 * ```json
 * {
 *  embedded = true
 * }
 * ```
 * @param content - The objet to be flattened
 *
 * @returns A single level object
 */
function defaultAppConfigTransform(content) {
    return Object.keys(content).reduce((result, key) => {
        const isObjectNotArray = content[key] !== null && typeof content[key] === 'object' && !Array.isArray(content[key]);
        return { ...result, ...(isObjectNotArray ? { ...content[key] } : { [key]: content[key] }) };
    }, {});
}
/**
 * Nest the content inside the first level objects expected by the local schema.
 * ```json
 * {
 *  embedded = true
 * }
 * ```
 * will be nested after applying the proper schema:
 * ```json
 * {
 *   pos: {
 *    embedded = true
 *   }
 * }
 * ```
 * @param content - The objet to be nested
 *
 * @returns The nested object
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function defaultAppConfigReverseTransform(schema, content) {
    return Object.keys(schema._def.shape()).reduce((result, key) => {
        let innerSchema = schema._def.shape()[key];
        if (innerSchema instanceof zod.ZodOptional) {
            innerSchema = innerSchema._def.innerType;
        }
        if (innerSchema instanceof zod.ZodObject) {
            result[key] = defaultAppConfigReverseTransform(innerSchema, content);
        }
        else {
            if (content[key] !== undefined)
                result[key] = content[key];
            delete content[key];
        }
        return result;
    }, {});
}
//# sourceMappingURL=specification.js.map