import { CachedAppInfo } from './local-storage.js';
import { PartnersSession } from './context/partner-account-info.js';
import { AppConfiguration, AppInterface } from '../models/app/app.js';
import { Identifiers, UuidOnlyIdentifiers } from '../models/app/identifiers.js';
import { OrganizationApp } from '../models/organization.js';
import { ExtensionInstance } from '../models/extensions/extension-instance.js';
import { ExtensionRegistration } from '../api/graphql/all_app_extension_registrations.js';
export declare const InvalidApiKeyErrorMessage: (apiKey: string) => {
    message: import("@shopify/cli-kit/node/output").TokenizedString;
    tryMessage: import("@shopify/cli-kit/node/output").TokenizedString;
};
export interface DevContextOptions {
    directory: string;
    apiKey?: string;
    storeFqdn?: string;
    reset: boolean;
}
interface DevContextOutput {
    remoteApp: Omit<OrganizationApp, 'apiSecretKeys'> & {
        apiSecret?: string;
    };
    remoteAppUpdated: boolean;
    storeFqdn: string;
    storeId: string;
    updateURLs: boolean | undefined;
    localApp: AppInterface;
}
/**
 * Make sure there is a valid context to execute `generate extension`
 *
 * We just need a valid app API key to access the Specifications API.
 * - If the API key is provided via flag, we use it.
 * - Else, if there is an API key in the current config or cache, we use it.
 * - Else, we prompt the user to select/create an app.
 *
 * The selection is then cached as the "dev" app for the current directory.
 */
export declare function ensureGenerateContext(options: {
    apiKey?: string;
    directory: string;
    reset: boolean;
    partnersSession: PartnersSession;
    configName?: string;
}): Promise<string>;
/**
 * Make sure there is a valid context to execute `dev`
 * That means we have a valid organization, app and dev store selected.
 *
 * If there are app/store from flags, we check if they are valid. If they are not, throw an error.
 * If there is info in the cache or current configuration, check if it is still valid and return it.
 * If there is no info (or is invalid):
 *  - Show prompts to select an org, app and dev store
 *  - The info will be updated in the cache or current configuration
 *
 * @param options - Current dev context options
 * @returns The selected org, app and dev store
 */
export declare function ensureDevContext(options: DevContextOptions, partnersSession: PartnersSession): Promise<DevContextOutput>;
export interface ReleaseContextOptions {
    app: AppInterface;
    apiKey?: string;
    reset: boolean;
    force: boolean;
}
interface ReleaseContextOutput {
    token: string;
    app: AppInterface;
    partnersApp: OrganizationApp;
}
interface DeployContextOutput {
    app: AppInterface;
    token: string;
    partnersApp: Omit<OrganizationApp, 'apiSecretKeys' | 'apiKey'>;
    identifiers: Identifiers;
    release: boolean;
}
export declare function ensureThemeExtensionDevContext(extension: ExtensionInstance, apiKey: string, token: string): Promise<ExtensionRegistration>;
export interface DeployContextOptions {
    app: AppInterface;
    apiKey?: string;
    reset: boolean;
    force: boolean;
    noRelease: boolean;
    commitReference?: string;
}
/**
 * Make sure there is a valid context to execute `deploy`
 * That means we have a valid session, organization and app.
 *
 * If there is an API key via flag, configuration or env file, we check if it is valid. Otherwise, throw an error.
 * If there is no API key (or is invalid), show prompts to select an org and app.
 * Finally, the info is updated in the env file.
 *
 * @param options - Current dev context options
 * @returns The selected org, app and dev store
 */
export declare function ensureDeployContext(options: DeployContextOptions): Promise<DeployContextOutput>;
export interface DraftExtensionsPushOptions {
    directory: string;
    apiKey?: string;
    reset: boolean;
    config?: string;
    enableDeveloperPreview: boolean;
}
export declare function ensureDraftExtensionsPushContext(draftExtensionsPushOptions: DraftExtensionsPushOptions): Promise<{
    app: AppInterface;
    partnersSession: PartnersSession;
    remoteExtensionIds: {
        [x: string]: string;
    };
    remoteApp: OrganizationApp;
}>;
/**
 * Make sure there is a valid context to execute `release`
 * That means we have a valid session, organization and app.
 *
 * If there is an API key via flag, configuration or env file, we check if it is valid. Otherwise, throw an error.
 * If there is no API key (or is invalid), show prompts to select an org and app.
 * If the app doesn't have the simplified deployments beta enabled, throw an error.
 * Finally, the info is updated in the env file.
 *
 * @param options - Current dev context options
 * @returns The selected org, app and dev store
 */
export declare function ensureReleaseContext(options: ReleaseContextOptions): Promise<ReleaseContextOutput>;
interface VersionListContextOptions {
    app: AppInterface;
    apiKey?: string;
    reset: false;
}
interface VersionsListContextOutput {
    partnersSession: PartnersSession;
    partnersApp: OrganizationApp;
}
/**
 * Make sure there is a valid context to execute `versions list`
 * That means we have a valid session, organization and app with the simplified deployments beta enabled.
 *
 * If there is an API key via flag, configuration or env file, we check if it is valid. Otherwise, throw an error.
 * If there is no API key (or is invalid), show prompts to select an org and app.
 * If the app doesn't have the simplified deployments beta enabled, throw an error.
 *
 * @param options - Current dev context options
 * @returns The partners token and app
 */
export declare function ensureVersionsListContext(options: VersionListContextOptions): Promise<VersionsListContextOutput>;
export declare function fetchOrCreateOrganizationApp(app: AppInterface, partnersSession: PartnersSession, directory?: string): Promise<OrganizationApp>;
export declare function fetchAppAndIdentifiers(options: {
    app: AppInterface;
    reset: boolean;
    apiKey?: string;
}, partnersSession: PartnersSession, reuseFromDev?: boolean): Promise<[OrganizationApp, Partial<UuidOnlyIdentifiers>]>;
export interface AppContext {
    configuration: AppConfiguration;
    cachedInfo?: CachedAppInfo;
    remoteApp?: OrganizationApp;
}
/**
 * Retrieve app info from the cache or the current configuration.
 *
 * @param reset - Whether to reset the cache or not.
 * @param directory - The directory containing the app.
 * @param token - The partners token.
 */
export declare function getAppContext({ reset, directory, partnersSession, configName, promptLinkingApp, }: {
    reset: boolean;
    directory: string;
    partnersSession: PartnersSession;
    configName?: string;
    promptLinkingApp?: boolean;
}): Promise<AppContext>;
interface CurrentlyUsedConfigInfoOptions {
    org: string;
    appName: string;
    devStore?: string;
    updateURLs?: string;
    configFile?: string;
    appDotEnv?: string;
    includeConfigOnDeploy?: boolean;
    resetMessage?: (string | {
        command: string;
    })[];
}
export declare function renderCurrentlyUsedConfigInfo({ org, appName, devStore, updateURLs, configFile, appDotEnv, resetMessage, includeConfigOnDeploy, }: CurrentlyUsedConfigInfoOptions): void;
export declare function logMetadataForLoadedContext(app: {
    organizationId: string;
    apiKey: string;
}): Promise<void>;
export declare function enableDeveloperPreview({ apiKey, token }: {
    apiKey: string;
    token: string;
}): Promise<boolean>;
export declare function disableDeveloperPreview({ apiKey, token }: {
    apiKey: string;
    token: string;
}): Promise<void>;
export declare function developerPreviewUpdate({ apiKey, token, enabled, }: {
    apiKey: string;
    token: string;
    enabled: boolean;
}): Promise<boolean>;
export {};
