import { manualMatchIds } from './id-manual-matching.js';
import { automaticMatchmaking } from './id-matching.js';
import { extensionMigrationPrompt, matchConfirmationPrompt } from './prompts.js';
import { createExtension } from '../dev/create-extension.js';
import { getUIExtensionsToMigrate, migrateExtensionsToUIExtension } from '../dev/migrate-to-ui-extension.js';
import { getFlowExtensionsToMigrate, migrateFlowExtensions } from '../dev/migrate-flow-extension.js';
import { ensureAuthenticatedPartners } from '@shopify/cli-kit/node/session';
import { outputCompleted } from '@shopify/cli-kit/node/output';
import { AbortSilentError } from '@shopify/cli-kit/node/error';
export async function ensureExtensionsIds(options, { extensionRegistrations: initialRemoteExtensions, dashboardManagedExtensionRegistrations: dashboardOnlyExtensions, }) {
    let remoteExtensions = initialRemoteExtensions;
    const validIdentifiers = options.envIdentifiers.extensions ?? {};
    const localExtensions = options.app.allExtensions.filter((ext) => ext.isUuidManaged());
    const uiExtensionsToMigrate = getUIExtensionsToMigrate(localExtensions, remoteExtensions, validIdentifiers);
    const flowExtensionsToMigrate = getFlowExtensionsToMigrate(localExtensions, dashboardOnlyExtensions, validIdentifiers);
    if (uiExtensionsToMigrate.length > 0) {
        const confirmedMigration = await extensionMigrationPrompt(uiExtensionsToMigrate);
        if (!confirmedMigration)
            throw new AbortSilentError();
        remoteExtensions = await migrateExtensionsToUIExtension(uiExtensionsToMigrate, options.appId, remoteExtensions);
    }
    if (flowExtensionsToMigrate.length > 0) {
        const confirmedMigration = await extensionMigrationPrompt(flowExtensionsToMigrate, false);
        if (!confirmedMigration)
            throw new AbortSilentError();
        const newRemoteExtensions = await migrateFlowExtensions(flowExtensionsToMigrate, options.appId, dashboardOnlyExtensions);
        remoteExtensions = remoteExtensions.concat(newRemoteExtensions);
    }
    const matchExtensions = await automaticMatchmaking(localExtensions, remoteExtensions, validIdentifiers, 'uuid');
    let validMatches = matchExtensions.identifiers;
    const extensionsToCreate = matchExtensions.toCreate ?? [];
    for (const pending of matchExtensions.toConfirm) {
        // eslint-disable-next-line no-await-in-loop
        const confirmed = await matchConfirmationPrompt(pending.local, pending.remote);
        if (confirmed) {
            validMatches[pending.local.localIdentifier] = pending.remote.uuid;
        }
        else {
            extensionsToCreate.push(pending.local);
        }
    }
    let onlyRemoteExtensions = matchExtensions.toManualMatch.remote ?? [];
    if (matchExtensions.toManualMatch.local.length > 0) {
        const matchResult = await manualMatchIds(matchExtensions.toManualMatch, 'uuid');
        validMatches = { ...validMatches, ...matchResult.identifiers };
        extensionsToCreate.push(...matchResult.toCreate);
        onlyRemoteExtensions = matchResult.onlyRemote;
    }
    return {
        validMatches,
        extensionsToCreate,
        dashboardOnlyExtensions,
    };
}
export async function deployConfirmed(options, extensionRegistrations, configurationRegistrations, { validMatches, extensionsToCreate, }) {
    const { extensionsNonUuidManaged, extensionsIdsNonUuidManaged } = await ensureNonUuidManagedExtensionsIds(configurationRegistrations, options.app, options.appId, options.includeDraftExtensions);
    const validMatchesById = {};
    if (extensionsToCreate.length > 0) {
        const newIdentifiers = await createExtensions(extensionsToCreate, options.appId);
        for (const [localIdentifier, registration] of Object.entries(newIdentifiers)) {
            validMatches[localIdentifier] = registration.uuid;
            validMatchesById[localIdentifier] = registration.id;
        }
    }
    // For extensions we also need the match by ID, not only UUID (doesn't apply to functions)
    for (const [localIdentifier, uuid] of Object.entries(validMatches)) {
        const registration = extensionRegistrations.find((registration) => registration.uuid === uuid);
        if (registration)
            validMatchesById[localIdentifier] = registration.id;
    }
    return {
        extensions: validMatches,
        extensionIds: { ...validMatchesById, ...extensionsIdsNonUuidManaged },
        extensionsNonUuidManaged,
    };
}
async function ensureNonUuidManagedExtensionsIds(remoteConfigurationRegistrations, app, appId, includeDraftExtensions = false) {
    let localExtensionRegistrations = includeDraftExtensions ? app.draftableExtensions : app.allExtensions;
    localExtensionRegistrations = localExtensionRegistrations.filter((ext) => !ext.isUuidManaged());
    const extensionsToCreate = [];
    const validMatches = {};
    const validMatchesById = {};
    localExtensionRegistrations.forEach((local) => {
        const possibleMatch = remoteConfigurationRegistrations.find((remote) => remote.type === local.graphQLType);
        if (possibleMatch) {
            validMatches[local.localIdentifier] = possibleMatch.uuid;
            validMatchesById[local.localIdentifier] = possibleMatch.id;
        }
        else
            extensionsToCreate.push(local);
    });
    if (extensionsToCreate.length > 0) {
        const newIdentifiers = await createExtensions(extensionsToCreate, appId, false);
        for (const [localIdentifier, registration] of Object.entries(newIdentifiers)) {
            validMatches[localIdentifier] = registration.uuid;
            validMatchesById[localIdentifier] = registration.id;
        }
    }
    return { extensionsNonUuidManaged: validMatches, extensionsIdsNonUuidManaged: validMatchesById };
}
async function createExtensions(extensions, appId, output = true) {
    const token = await ensureAuthenticatedPartners();
    const result = {};
    for (const extension of extensions) {
        // Create one at a time to avoid API rate limiting issues.
        // eslint-disable-next-line no-await-in-loop
        const registration = await createExtension(appId, extension.graphQLType, extension.handle, token, extension.contextValue);
        if (output)
            outputCompleted(`Created extension ${extension.handle}.`);
        result[extension.localIdentifier] = registration;
    }
    return result;
}
//# sourceMappingURL=identifiers-extensions.js.map