/* eslint-disable require-atomic-updates */
import { uploadThemeExtensions, uploadExtensionsBundle } from './deploy/upload.js';
import { ensureDeployContext } from './context.js';
import { bundleAndBuildExtensions } from './deploy/bundle.js';
import { updateAppIdentifiers } from '../models/app/identifiers.js';
import { renderInfo, renderSuccess, renderTasks } from '@shopify/cli-kit/node/ui';
import { inTemporaryDirectory, mkdir } from '@shopify/cli-kit/node/fs';
import { joinPath, dirname } from '@shopify/cli-kit/node/path';
import { outputNewline, outputInfo, formatPackageManagerCommand } from '@shopify/cli-kit/node/output';
import { useThemebundling } from '@shopify/cli-kit/node/context/local';
import { getArrayRejectingUndefined } from '@shopify/cli-kit/common/array';
export async function deploy(options) {
    // eslint-disable-next-line prefer-const
    let { app, identifiers, partnersApp, token, release } = await ensureDeployContext(options);
    const apiKey = identifiers.app;
    outputNewline();
    if (release) {
        outputInfo(`Releasing a new app version as part of ${partnersApp.title}`);
    }
    else {
        outputInfo(`Creating a new app version as part of ${partnersApp.title}`);
    }
    outputNewline();
    let uploadExtensionsBundleResult;
    await inTemporaryDirectory(async (tmpDir) => {
        try {
            const bundle = app.allExtensions.some((ext) => ext.features.includes('bundling'));
            let bundlePath;
            if (bundle) {
                bundlePath = joinPath(tmpDir, `bundle.zip`);
                await mkdir(dirname(bundlePath));
            }
            await bundleAndBuildExtensions({ app, bundlePath, identifiers });
            let uploadTaskTitle;
            if (release) {
                uploadTaskTitle = 'Releasing an app version';
            }
            else {
                uploadTaskTitle = 'Creating an app version';
            }
            const tasks = [
                {
                    title: 'Running validation',
                    task: async () => {
                        await app.preDeployValidation();
                    },
                },
                {
                    title: uploadTaskTitle,
                    task: async () => {
                        const appModules = await Promise.all(app.allExtensions.flatMap((ext) => ext.bundleConfig({ identifiers, token, apiKey })));
                        uploadExtensionsBundleResult = await uploadExtensionsBundle({
                            apiKey,
                            bundlePath,
                            appModules: getArrayRejectingUndefined(appModules),
                            release,
                            token,
                            extensionIds: identifiers.extensionIds,
                            message: options.message,
                            version: options.version,
                            commitReference: options.commitReference,
                        });
                        if (!useThemebundling()) {
                            const themeExtensions = app.allExtensions.filter((ext) => ext.isThemeExtension);
                            await uploadThemeExtensions(themeExtensions, { apiKey, identifiers, token });
                        }
                        app = await updateAppIdentifiers({ app, identifiers, command: 'deploy' });
                    },
                },
            ];
            await renderTasks(tasks);
            await outputCompletionMessage({
                app,
                release,
                uploadExtensionsBundleResult,
            });
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
        }
        catch (error) {
            /**
             * If deployment fails when uploading we want the identifiers to be persisted
             * for the next run.
             */
            await updateAppIdentifiers({ app, identifiers, command: 'deploy' });
            throw error;
        }
    });
}
async function outputCompletionMessage({ app, release, uploadExtensionsBundleResult, }) {
    const linkAndMessage = [
        { link: { label: uploadExtensionsBundleResult.versionTag, url: uploadExtensionsBundleResult.location } },
        uploadExtensionsBundleResult.message ? `\n${uploadExtensionsBundleResult.message}` : '',
    ];
    if (release) {
        return uploadExtensionsBundleResult.deployError
            ? renderInfo({
                headline: 'New version created, but not released.',
                body: [...linkAndMessage, `\n\n${uploadExtensionsBundleResult.deployError}`],
            })
            : renderSuccess({
                headline: 'New version released to users.',
                body: linkAndMessage,
            });
    }
    return renderSuccess({
        headline: 'New version created.',
        body: linkAndMessage,
        nextSteps: [
            [
                'Run',
                {
                    command: formatPackageManagerCommand(app.packageManager, 'shopify app release', `--version=${uploadExtensionsBundleResult.versionTag}`),
                },
                'to release this version to users.',
            ],
        ],
    });
}
//# sourceMappingURL=deploy.js.map