import { generateFrontendURL, generatePartnersURLs, getURLs, shouldOrPromptUpdateURLs, startTunnelPlugin, updateURLs, } from './dev/urls.js';
import { ensureDevContext, enableDeveloperPreview, disableDeveloperPreview, developerPreviewUpdate } from './context.js';
import { fetchAppPreviewMode } from './dev/fetch.js';
import { installAppDependencies } from './dependencies.js';
import { setupDevProcesses } from './dev/processes/setup-dev-processes.js';
import { frontAndBackendConfig } from './dev/processes/utils.js';
import { outputUpdateURLsResult, renderDev } from './dev/ui.js';
import { setCachedAppInfo } from './local-storage.js';
import { canEnablePreviewMode } from './extensions/common.js';
import { fetchPartnersSession } from './context/partner-account-info.js';
import { isCurrentAppSchema, getAppScopesArray } from '../models/app/app.js';
import { getAnalyticsTunnelType } from '../utilities/analytics.js';
import { ports } from '../constants.js';
import metadata from '../metadata.js';
import { performActionWithRetryAfterRecovery } from '@shopify/cli-kit/common/retry';
import { AbortController } from '@shopify/cli-kit/node/abort';
import { checkPortAvailability, getAvailableTCPPort } from '@shopify/cli-kit/node/tcp';
import { getBackendPort } from '@shopify/cli-kit/node/environment';
import { basename } from '@shopify/cli-kit/node/path';
import { renderWarning } from '@shopify/cli-kit/node/ui';
import { reportAnalyticsEvent } from '@shopify/cli-kit/node/analytics';
import { formatPackageManagerCommand, outputDebug } from '@shopify/cli-kit/node/output';
import { hashString } from '@shopify/cli-kit/node/crypto';
import { AbortError } from '@shopify/cli-kit/node/error';
import { ensureAuthenticatedPartners } from '@shopify/cli-kit/node/session';
export async function dev(commandOptions) {
    const config = await prepareForDev(commandOptions);
    await actionsBeforeSettingUpDevProcesses(config);
    const { processes, graphiqlUrl, previewUrl } = await setupDevProcesses(config);
    await actionsBeforeLaunchingDevProcesses(config);
    await launchDevProcesses({ processes, previewUrl, graphiqlUrl, config });
}
async function prepareForDev(commandOptions) {
    // Be optimistic about tunnel creation and do it as early as possible
    const tunnelPort = await getAvailableTCPPort();
    let tunnelClient;
    if (!commandOptions.tunnelUrl && !commandOptions.noTunnel) {
        tunnelClient = await startTunnelPlugin(commandOptions.commandConfig, tunnelPort, 'cloudflare');
    }
    const partnersSession = await fetchPartnersSession();
    const token = partnersSession.token;
    const { storeFqdn, storeId, remoteApp, remoteAppUpdated, updateURLs: cachedUpdateURLs, localApp: app, } = await ensureDevContext(commandOptions, partnersSession);
    const apiKey = remoteApp.apiKey;
    let localApp = app;
    if (!commandOptions.skipDependenciesInstallation && !localApp.usesWorkspaces) {
        localApp = await installAppDependencies(localApp);
    }
    const graphiqlPort = commandOptions.graphiqlPort || ports.graphiql;
    const { graphiqlKey } = commandOptions;
    const { webs, ...network } = await setupNetworkingOptions(localApp.webs, graphiqlPort, apiKey, token, {
        noTunnel: commandOptions.noTunnel,
        tunnelUrl: commandOptions.tunnelUrl,
    }, tunnelClient);
    localApp.webs = webs;
    const partnerUrlsUpdated = await handleUpdatingOfPartnerUrls(webs, commandOptions.update, network, localApp, cachedUpdateURLs, remoteApp, apiKey, token);
    return {
        storeFqdn,
        storeId,
        remoteApp,
        remoteAppUpdated,
        localApp,
        token,
        commandOptions,
        network,
        partnerUrlsUpdated,
        graphiqlPort,
        graphiqlKey,
    };
}
async function actionsBeforeSettingUpDevProcesses({ localApp, remoteApp }) {
    if (isCurrentAppSchema(localApp.configuration) &&
        !localApp.configuration.access_scopes?.use_legacy_install_flow &&
        localApp.configuration.access_scopes?.scopes !== remoteApp.requestedAccessScopes?.join(',')) {
        const nextSteps = [
            [
                'Run',
                { command: formatPackageManagerCommand(localApp.packageManager, 'shopify app deploy') },
                'to push your scopes to the Partner Dashboard',
            ],
        ];
        renderWarning({
            headline: [`The scopes in your TOML don't match the scopes in your Partner Dashboard`],
            body: [
                `Scopes in ${basename(localApp.configuration.path)}:`,
                scopesMessage(getAppScopesArray(localApp.configuration)),
                '\n',
                'Scopes in Partner Dashboard:',
                scopesMessage(remoteApp.requestedAccessScopes || []),
            ],
            nextSteps,
        });
    }
}
async function actionsBeforeLaunchingDevProcesses(config) {
    setPreviousAppId(config.commandOptions.directory, config.remoteApp.apiKey);
    await logMetadataForDev({
        devOptions: config.commandOptions,
        tunnelUrl: config.network.proxyUrl,
        shouldUpdateURLs: config.partnerUrlsUpdated,
        storeFqdn: config.storeFqdn,
    });
    await reportAnalyticsEvent({ config: config.commandOptions.commandConfig, exitMode: 'ok' });
}
async function handleUpdatingOfPartnerUrls(webs, commandSpecifiedToUpdate, network, localApp, cachedUpdateURLs, remoteApp, apiKey, token) {
    const { backendConfig, frontendConfig } = frontAndBackendConfig(webs);
    let shouldUpdateURLs = false;
    if (frontendConfig || backendConfig) {
        if (commandSpecifiedToUpdate) {
            const newURLs = generatePartnersURLs(network.proxyUrl, webs.map(({ configuration }) => configuration.auth_callback_path).find((path) => path), isCurrentAppSchema(localApp.configuration) ? localApp.configuration.app_proxy : undefined);
            shouldUpdateURLs = await shouldOrPromptUpdateURLs({
                currentURLs: network.currentUrls,
                appDirectory: localApp.directory,
                cachedUpdateURLs,
                newApp: remoteApp.newApp,
                localApp,
                apiKey,
            });
            if (shouldUpdateURLs)
                await updateURLs(newURLs, apiKey, token, localApp);
            await outputUpdateURLsResult(shouldUpdateURLs, newURLs, remoteApp, localApp);
        }
    }
    return shouldUpdateURLs;
}
async function setupNetworkingOptions(webs, graphiqlPort, apiKey, token, frontEndOptions, tunnelClient) {
    const { backendConfig, frontendConfig } = frontAndBackendConfig(webs);
    await validateCustomPorts(webs, graphiqlPort);
    // generateFrontendURL still uses the old naming of frontendUrl and frontendPort,
    // we can rename them to proxyUrl and proxyPort when we delete dev.ts
    const [{ frontendUrl, frontendPort: proxyPort, usingLocalhost }, backendPort, currentUrls] = await Promise.all([
        generateFrontendURL({
            ...frontEndOptions,
            tunnelClient,
        }),
        getBackendPort() || backendConfig?.configuration.port || getAvailableTCPPort(),
        getURLs(apiKey, token),
    ]);
    const proxyUrl = usingLocalhost ? `${frontendUrl}:${proxyPort}` : frontendUrl;
    let frontendPort = frontendConfig?.configuration.port;
    if (frontendConfig) {
        if (!frontendPort) {
            frontendPort = frontendConfig === backendConfig ? backendPort : await getAvailableTCPPort();
        }
        frontendConfig.configuration.port = frontendPort;
    }
    frontendPort = frontendPort ?? (await getAvailableTCPPort());
    return {
        proxyUrl,
        proxyPort,
        frontendPort,
        backendPort,
        currentUrls,
        webs,
    };
}
async function launchDevProcesses({ processes, previewUrl, graphiqlUrl, config, }) {
    const abortController = new AbortController();
    const processesForTaskRunner = processes.map((process) => {
        const outputProcess = {
            prefix: process.prefix,
            action: async (stdout, stderr, signal) => {
                const fn = process.function;
                return fn({ stdout, stderr, abortSignal: signal }, process.options);
            },
        };
        return outputProcess;
    });
    const apiKey = config.remoteApp.apiKey;
    const token = config.token;
    const app = {
        canEnablePreviewMode: await canEnablePreviewMode({
            remoteApp: config.remoteApp,
            localApp: config.localApp,
            token,
            apiKey,
        }),
        developmentStorePreviewEnabled: config.remoteApp.developmentStorePreviewEnabled,
        apiKey,
        token,
    };
    return renderDev({
        processes: processesForTaskRunner,
        previewUrl,
        graphiqlUrl,
        graphiqlPort: config.graphiqlPort,
        app,
        abortController,
        developerPreview: developerPreviewController(apiKey, token),
    });
}
export function developerPreviewController(apiKey, originalToken) {
    let currentToken = originalToken;
    const refreshToken = async () => {
        const newToken = await ensureAuthenticatedPartners([], process.env, { noPrompt: true });
        if (newToken)
            currentToken = newToken;
    };
    const withRefreshToken = async (fn) => {
        try {
            const result = await performActionWithRetryAfterRecovery(async () => fn(currentToken), refreshToken);
            return result;
        }
        catch (err) {
            outputDebug('Failed to refresh token');
            throw err;
        }
    };
    return {
        fetchMode: async () => withRefreshToken(async (token) => Boolean(await fetchAppPreviewMode(apiKey, token))),
        enable: async () => withRefreshToken(async (token) => {
            await enableDeveloperPreview({ apiKey, token });
        }),
        disable: async () => withRefreshToken(async (token) => {
            await disableDeveloperPreview({ apiKey, token });
        }),
        update: async (state) => withRefreshToken(async (token) => developerPreviewUpdate({ apiKey, token, enabled: state })),
    };
}
export async function logMetadataForDev(options) {
    const tunnelType = await getAnalyticsTunnelType(options.devOptions.commandConfig, options.tunnelUrl);
    await metadata.addPublicMetadata(() => ({
        cmd_dev_tunnel_type: tunnelType,
        cmd_dev_tunnel_custom_hash: tunnelType === 'custom' ? hashString(options.tunnelUrl) : undefined,
        cmd_dev_urls_updated: options.shouldUpdateURLs,
        store_fqdn_hash: hashString(options.storeFqdn),
        cmd_app_dependency_installation_skipped: options.devOptions.skipDependenciesInstallation,
        cmd_app_reset_used: options.devOptions.reset,
    }));
    await metadata.addSensitiveMetadata(() => ({
        store_fqdn: options.storeFqdn,
        cmd_dev_tunnel_custom: tunnelType === 'custom' ? options.tunnelUrl : undefined,
    }));
}
export function scopesMessage(scopes) {
    return {
        list: {
            items: scopes.length === 0 ? ['No scopes'] : scopes,
        },
    };
}
export async function validateCustomPorts(webConfigs, graphiqlPort) {
    const allPorts = webConfigs.map((config) => config.configuration.port).filter((port) => port);
    const duplicatedPort = allPorts.find((port, index) => allPorts.indexOf(port) !== index);
    if (duplicatedPort) {
        throw new AbortError(`Found port ${duplicatedPort} for multiple webs.`, 'Please define a unique port for each web.');
    }
    await Promise.all([
        ...allPorts.map(async (port) => {
            const portAvailable = await checkPortAvailability(port);
            if (!portAvailable) {
                throw new AbortError(`Hard-coded port ${port} is not available, please choose a different one.`);
            }
        }),
        (async () => {
            const portAvailable = await checkPortAvailability(graphiqlPort);
            if (!portAvailable) {
                const errorMessage = `Port ${graphiqlPort} is not available for serving GraphiQL.`;
                const tryMessage = ['Choose a different port by setting the', { command: '--graphiql-port' }, 'flag.'];
                throw new AbortError(errorMessage, tryMessage);
            }
        })(),
    ]);
}
export function setPreviousAppId(directory, apiKey) {
    setCachedAppInfo({ directory, previousAppId: apiKey });
}
//# sourceMappingURL=dev.js.map