import { AllAppExtensionRegistrationsQuery, } from '../../api/graphql/all_app_extension_registrations.js';
import { AllOrganizationsQuery } from '../../api/graphql/all_orgs.js';
import { FindOrganizationQuery } from '../../api/graphql/find_org.js';
import { FindAppQuery } from '../../api/graphql/find_app.js';
import { FindAppPreviewModeQuery } from '../../api/graphql/find_app_preview_mode.js';
import { FindOrganizationBasicQuery } from '../../api/graphql/find_org_basic.js';
import { AllDevStoresByOrganizationQuery, } from '../../api/graphql/all_dev_stores_by_org.js';
import { FindStoreByDomainQuery } from '../../api/graphql/find_store_by_domain.js';
import { ActiveAppVersionQuery } from '../../api/graphql/app_active_version.js';
import { isServiceAccount, isUserAccount } from '../context/partner-account-info.js';
import { partnersRequest } from '@shopify/cli-kit/node/api/partners';
import { AbortError } from '@shopify/cli-kit/node/error';
import { outputContent, outputToken } from '@shopify/cli-kit/node/output';
export class NoOrgError extends AbortError {
    constructor(partnersAccount, organizationId) {
        let accountIdentifier = 'unknown';
        let identifierMessage = (formattedIdentifier) => `an ${formattedIdentifier}`;
        if (isServiceAccount(partnersAccount)) {
            accountIdentifier = partnersAccount.orgName;
            identifierMessage = (formattedIdentifier) => `the ${formattedIdentifier} organization`;
        }
        else if (isUserAccount(partnersAccount)) {
            accountIdentifier = partnersAccount.email;
            identifierMessage = (formattedIdentifier) => `the ${formattedIdentifier} user`;
        }
        const formattedIdentifier = outputContent `${outputToken.yellow(accountIdentifier)}`.value;
        const nextSteps = [
            [
                `Your current active session is associated with ${identifierMessage(formattedIdentifier)} account. To start a new session with a different account, run`,
                { command: 'shopify auth logout' },
            ],
            [
                'Have you',
                {
                    link: {
                        label: 'created a Shopify Partners organization',
                        url: 'https://partners.shopify.com/signup',
                    },
                },
                {
                    char: '?',
                },
            ],
            [
                'Does your account include',
                {
                    subdued: 'Manage app',
                },
                'permissions?, please contact the owner of the organization to grant you access.',
            ],
            'Have you confirmed your accounts from the emails you received?',
            [
                'Need to connect to a different App or organization? Run the command again with',
                {
                    command: '--reset',
                },
            ],
        ];
        if (organizationId) {
            nextSteps.push([
                'Do you have access to the right Shopify Partners organization? The CLI is loading',
                { link: { label: 'this organization', url: `https://partner.shopify.com/${organizationId}` } },
            ]);
        }
        super(`No Organization found`, undefined, nextSteps);
    }
}
export async function fetchAppExtensionRegistrations({ token, apiKey, }) {
    const query = AllAppExtensionRegistrationsQuery;
    const result = await partnersRequest(query, token, {
        apiKey,
    });
    return result;
}
export async function fetchActiveAppVersion({ token, apiKey, }) {
    const query = ActiveAppVersionQuery;
    const result = await partnersRequest(query, token, {
        apiKey,
    });
    return result;
}
/**
 * Fetch all organizations the user belongs to
 * If the user doesn't belong to any org, throw an error
 * @param token - Token to access partners API
 * @returns List of organizations
 */
export async function fetchOrganizations(partnersSession) {
    const query = AllOrganizationsQuery;
    const result = await partnersRequest(query, partnersSession.token);
    const organizations = result.organizations.nodes;
    if (organizations.length === 0)
        throw new NoOrgError(partnersSession.accountInfo);
    return organizations;
}
/**
 * Fetch all apps and stores for the given organization
 * @param orgId - Organization ID
 * @param token - Token to access partners API
 * @returns Current organization details and list of apps and stores
 */
export async function fetchOrgAndApps(orgId, partnersSession, title) {
    const query = FindOrganizationQuery;
    const params = { id: orgId };
    if (title)
        params.title = title;
    const result = await partnersRequest(query, partnersSession.token, params);
    const org = result.organizations.nodes[0];
    if (!org)
        throw new NoOrgError(partnersSession.accountInfo, orgId);
    const parsedOrg = { id: org.id, businessName: org.businessName };
    return { organization: parsedOrg, apps: org.apps, stores: [] };
}
export var BetaFlag;
(function (BetaFlag) {
})(BetaFlag || (BetaFlag = {}));
const FlagMap = {};
export async function fetchAppDetailsFromApiKey(apiKey, token) {
    const res = await partnersRequest(FindAppQuery, token, {
        apiKey,
    });
    const app = res.app;
    if (app) {
        const defaultActiveBetas = [];
        const remoteDisabledFlags = app.disabledBetas.map((flag) => FlagMap[flag]);
        const betas = defaultActiveBetas.filter((beta) => !remoteDisabledFlags.includes(beta));
        return { ...app, betas };
    }
}
export async function fetchAppPreviewMode(apiKey, token) {
    const res = await partnersRequest(FindAppPreviewModeQuery, token, {
        apiKey,
    });
    return res.app?.developmentStorePreviewEnabled;
}
export async function fetchOrgFromId(id, partnersSession) {
    const query = FindOrganizationBasicQuery;
    const res = await partnersRequest(query, partnersSession.token, { id });
    const org = res.organizations.nodes[0];
    if (!org)
        throw new NoOrgError(partnersSession.accountInfo, id);
    return org;
}
export async function fetchAllDevStores(orgId, token) {
    const query = AllDevStoresByOrganizationQuery;
    const result = await partnersRequest(query, token, {
        id: orgId,
    });
    return result.organizations.nodes[0].stores.nodes;
}
/**
 * Returns the organization and the store based on passed domain
 * If a store with that domain doesn't exist the method returns undefined
 * @param orgId - Organization ID
 * @param token - Token to access partners API
 * @param shopDomain - shop domain fqdn
 */
export async function fetchStoreByDomain(orgId, token, shopDomain) {
    const query = FindStoreByDomainQuery;
    const result = await partnersRequest(query, token, {
        id: orgId,
        shopDomain,
    });
    const org = result.organizations.nodes[0];
    if (!org) {
        return undefined;
    }
    const parsedOrg = { id: org.id, businessName: org.businessName };
    const store = org.stores.nodes[0];
    return { organization: parsedOrg, store };
}
//# sourceMappingURL=fetch.js.map