import metadata from '../../../../metadata.js';
import { ConcurrentOutput } from '@shopify/cli-kit/node/ui/components';
import { useAbortSignal } from '@shopify/cli-kit/node/ui/hooks';
import React, { useEffect, useMemo, useRef, useState } from 'react';
import { Box, Text, useInput, useStdin } from 'ink';
import { handleCtrlC } from '@shopify/cli-kit/node/ui';
import { openURL } from '@shopify/cli-kit/node/system';
import figures from '@shopify/cli-kit/node/figures';
import { isUnitTest } from '@shopify/cli-kit/node/context/local';
import { treeKill } from '@shopify/cli-kit/node/tree-kill';
const Dev = ({ abortController, processes, previewUrl, graphiqlUrl = '', graphiqlPort, app, pollingTime = 5000, developerPreview, isEditionWeek, }) => {
    const { canEnablePreviewMode, developmentStorePreviewEnabled } = app;
    const { isRawModeSupported: canUseShortcuts } = useStdin();
    const pollingInterval = useRef();
    const localhostGraphiqlUrl = `http://localhost:${graphiqlPort}/graphiql`;
    const defaultStatusMessage = `Preview URL: ${previewUrl}${graphiqlUrl ? `\nGraphiQL URL: ${localhostGraphiqlUrl}` : ''}`;
    const [statusMessage, setStatusMessage] = useState(defaultStatusMessage);
    const { isAborted } = useAbortSignal(abortController.signal, async (err) => {
        if (err) {
            setStatusMessage('Shutting down dev because of an error ...');
        }
        else {
            setStatusMessage('Shutting down dev ...');
            setTimeout(() => {
                if (isUnitTest())
                    return;
                treeKill(process.pid, 'SIGINT', false, () => {
                    process.exit(0);
                });
            }, 2000);
        }
        clearInterval(pollingInterval.current);
        await developerPreview.disable();
    });
    const [devPreviewEnabled, setDevPreviewEnabled] = useState(true);
    const [error, setError] = useState(undefined);
    const errorHandledProcesses = useMemo(() => {
        return processes.map((process) => {
            return {
                ...process,
                action: async (stdout, stderr, signal) => {
                    try {
                        return await process.action(stdout, stderr, signal);
                        // eslint-disable-next-line no-catch-all/no-catch-all
                    }
                    catch (error) {
                        abortController.abort(error);
                    }
                },
            };
        });
    }, [processes, abortController]);
    useEffect(() => {
        const pollDevPreviewMode = async () => {
            try {
                const enabled = await developerPreview.fetchMode();
                setDevPreviewEnabled(enabled ?? false);
                setError('');
                // eslint-disable-next-line no-catch-all/no-catch-all
            }
            catch (_) {
                setError('Failed to fetch the latest status of the development store preview, trying again in 5 seconds.');
            }
        };
        const enablePreviewMode = async () => {
            // Enable dev preview on app dev start
            try {
                await developerPreview.enable();
                setError('');
                // eslint-disable-next-line no-catch-all/no-catch-all
            }
            catch (_) {
                setError('Failed to turn on development store preview automatically.\nTry turning it on manually by pressing `d`.');
                setDevPreviewEnabled(Boolean(developmentStorePreviewEnabled));
            }
        };
        if (canEnablePreviewMode) {
            // eslint-disable-next-line @typescript-eslint/no-floating-promises
            enablePreviewMode();
            const startPolling = () => {
                return setInterval(
                // eslint-disable-next-line @typescript-eslint/no-misused-promises
                () => pollDevPreviewMode(), pollingTime);
            };
            pollingInterval.current = startPolling();
        }
        return () => {
            clearInterval(pollingInterval.current);
        };
    }, [canEnablePreviewMode]);
    useInput((input, key) => {
        handleCtrlC(input, key, () => abortController.abort());
        const onInput = async () => {
            try {
                setError('');
                if (input === 'p' && previewUrl) {
                    await metadata.addPublicMetadata(() => ({
                        cmd_dev_preview_url_opened: true,
                    }));
                    await openURL(previewUrl);
                }
                else if (input === 'g' && graphiqlUrl) {
                    await metadata.addPublicMetadata(() => ({
                        cmd_dev_graphiql_opened: true,
                    }));
                    await openURL(localhostGraphiqlUrl);
                }
                else if (input === 'q') {
                    abortController.abort();
                }
                else if (input === 'e' && isEditionWeek) {
                    await openURL('https://shopify.link/yQmk');
                }
                else if (input === 'd' && canEnablePreviewMode) {
                    await metadata.addPublicMetadata(() => ({
                        cmd_dev_dev_preview_toggle_used: true,
                    }));
                    const newDevPreviewEnabled = !devPreviewEnabled;
                    setDevPreviewEnabled(newDevPreviewEnabled);
                    try {
                        const developerPreviewUpdateSucceded = await developerPreview.update(newDevPreviewEnabled);
                        if (!developerPreviewUpdateSucceded) {
                            throw new Error(`Failed to turn ${newDevPreviewEnabled ? 'on' : 'off'} development store preview.`);
                        }
                        // eslint-disable-next-line no-catch-all/no-catch-all
                    }
                    catch (_) {
                        setDevPreviewEnabled(devPreviewEnabled);
                        setError(`Failed to turn ${newDevPreviewEnabled ? 'on' : 'off'} development store preview.`);
                    }
                }
                // eslint-disable-next-line no-catch-all/no-catch-all
            }
            catch (_) {
                setError('Failed to handle your input.');
            }
        };
        // eslint-disable-next-line @typescript-eslint/no-floating-promises
        onInput();
    }, { isActive: Boolean(canUseShortcuts) });
    const now = new Date();
    const season = now.getMonth() > 3 ? 'Summer' : 'Winter';
    const year = now.getFullYear();
    return (React.createElement(React.Fragment, null,
        React.createElement(ConcurrentOutput, { processes: errorHandledProcesses, abortSignal: abortController.signal, keepRunningAfterProcessesResolve: true }),
        !isAborted ? (React.createElement(Box, { marginY: 1, paddingTop: 1, flexDirection: "column", flexGrow: 1, borderStyle: "single", borderBottom: false, borderLeft: false, borderRight: false, borderTop: true },
            canUseShortcuts ? (React.createElement(Box, { flexDirection: "column" },
                isEditionWeek ? (React.createElement(Text, null,
                    figures.pointerSmall,
                    " Press ",
                    React.createElement(Text, { bold: true }, "e"),
                    " ",
                    figures.lineVertical,
                    " check out ",
                    season,
                    " Edition",
                    ` ${year}`,
                    ", live NOW with 100+ product announcements!")) : null,
                canEnablePreviewMode ? (React.createElement(Text, null,
                    figures.pointerSmall,
                    " Press ",
                    React.createElement(Text, { bold: true }, "d"),
                    " ",
                    figures.lineVertical,
                    " toggle development store preview: ",
                    devPreviewEnabled ? React.createElement(Text, { color: "green" }, "\u2714 on") : React.createElement(Text, { color: "red" }, "\u2716 off"))) : null,
                graphiqlUrl ? (React.createElement(Text, null,
                    figures.pointerSmall,
                    " Press ",
                    React.createElement(Text, { bold: true }, "g"),
                    " ",
                    figures.lineVertical,
                    " open GraphiQL (Admin API) in your browser")) : null,
                React.createElement(Text, null,
                    figures.pointerSmall,
                    " Press ",
                    React.createElement(Text, { bold: true }, "p"),
                    " ",
                    figures.lineVertical,
                    " preview in your browser"),
                React.createElement(Text, null,
                    figures.pointerSmall,
                    " Press ",
                    React.createElement(Text, { bold: true }, "q"),
                    " ",
                    figures.lineVertical,
                    " quit"))) : null,
            React.createElement(Box, { marginTop: canUseShortcuts ? 1 : 0 },
                React.createElement(Text, null, statusMessage)),
            error ? React.createElement(Text, { color: "red" }, error) : null)) : null));
};
export { Dev };
//# sourceMappingURL=Dev.js.map