import { fetchOrCreateOrganizationApp } from './context.js';
import { fetchPartnersSession } from './context/partner-account-info.js';
import { getAppIdentifiers } from '../models/app/identifiers.js';
import { ApiSchemaDefinitionQuery, } from '../api/graphql/functions/api_schema_definition.js';
import { TargetSchemaDefinitionQuery, } from '../api/graphql/functions/target_schema_definition.js';
import { partnersRequest } from '@shopify/cli-kit/node/api/partners';
import { isTerminalInteractive } from '@shopify/cli-kit/node/context/local';
import { AbortError } from '@shopify/cli-kit/node/error';
import { outputContent, outputInfo } from '@shopify/cli-kit/node/output';
import { writeFile } from '@shopify/cli-kit/node/fs';
import { joinPath } from '@shopify/cli-kit/node/path';
export async function generateSchemaService(options) {
    const { extension, app } = options;
    const partnersSession = await fetchPartnersSession();
    const token = partnersSession.token;
    const { api_version: version, type, targeting } = extension.configuration;
    let apiKey = options.apiKey || getAppIdentifiers({ app }).app;
    const stdout = options.stdout;
    if (!apiKey) {
        if (!isTerminalInteractive()) {
            throw new AbortError(outputContent `No Client ID was provided.`, outputContent `Provide a Client ID with the --client-id flag.`);
        }
        apiKey = (await fetchOrCreateOrganizationApp(app, partnersSession)).apiKey;
    }
    const usingTargets = Boolean(targeting?.length);
    const definition = await (usingTargets
        ? generateSchemaFromTarget({
            localIdentifier: extension.localIdentifier,
            token,
            apiKey,
            target: targeting[0].target,
            version,
        })
        : generateSchemaFromType({ localIdentifier: extension.localIdentifier, token, apiKey, type, version }));
    if (stdout) {
        outputInfo(definition);
    }
    else {
        const outputPath = joinPath(options.path, 'schema.graphql');
        await writeFile(outputPath, definition);
        outputInfo(`GraphQL Schema for ${extension.localIdentifier} written to ${outputPath}`);
    }
}
async function generateSchemaFromTarget({ localIdentifier, token, apiKey, target, version, }) {
    const query = TargetSchemaDefinitionQuery;
    const variables = {
        apiKey,
        target,
        version,
    };
    const response = await partnersRequest(query, token, variables);
    if (!response.definition) {
        throw new AbortError(outputContent `A schema could not be generated for ${localIdentifier}`, outputContent `Check that the Function targets and version are valid.`);
    }
    return response.definition;
}
async function generateSchemaFromType({ localIdentifier, token, apiKey, version, type, }) {
    const query = ApiSchemaDefinitionQuery;
    const variables = {
        apiKey,
        version,
        type,
    };
    const response = await partnersRequest(query, token, variables);
    if (!response.definition) {
        throw new AbortError(outputContent `A schema could not be generated for ${localIdentifier}`, outputContent `Check that the Function API type and version are valid.`);
    }
    return response.definition;
}
//# sourceMappingURL=generate-schema.js.map