import { outputEnv } from './app/env/show.js';
import { getAppContext } from './context.js';
import { fetchPartnersSession, isServiceAccount, isUserAccount } from './context/partner-account-info.js';
import { getAppScopes } from '../models/app/app.js';
import { configurationFileNames } from '../constants.js';
import { platformAndArch } from '@shopify/cli-kit/node/os';
import { checkForNewVersion } from '@shopify/cli-kit/node/node-package-manager';
import { linesToColumns } from '@shopify/cli-kit/common/string';
import { relativePath } from '@shopify/cli-kit/node/path';
import { outputContent, outputToken, formatSection, stringifyMessage, getOutputUpdateCLIReminder, } from '@shopify/cli-kit/node/output';
export async function info(app, options) {
    if (options.webEnv) {
        return infoWeb(app, options);
    }
    else {
        return infoApp(app, options);
    }
}
export async function infoWeb(app, { format }) {
    return outputEnv(app, format);
}
export async function infoApp(app, options) {
    if (options.format === 'json') {
        const appWithSupportedExtensions = {
            ...app,
            allExtensions: app.allExtensions.filter((ext) => ext.isReturnedAsInfo()),
        };
        return outputContent `${JSON.stringify(appWithSupportedExtensions, null, 2)}`;
    }
    else {
        const appInfo = new AppInfo(app, options);
        return appInfo.output();
    }
}
const UNKNOWN_TEXT = outputContent `${outputToken.italic('unknown')}`.value;
const NOT_CONFIGURED_TEXT = outputContent `${outputToken.italic('Not yet configured')}`.value;
class AppInfo {
    constructor(app, options) {
        this.app = app;
        this.options = options;
    }
    async output() {
        const sections = [
            await this.devConfigsSection(),
            this.projectSettingsSection(),
            await this.appComponentsSection(),
            await this.systemInfoSection(),
        ];
        return sections.map((sectionContents) => formatSection(...sectionContents)).join('\n\n');
    }
    async devConfigsSection() {
        const title = `Current app configuration`;
        const partnersSession = await fetchPartnersSession();
        const { cachedInfo } = await getAppContext({
            partnersSession,
            directory: this.app.directory,
            reset: false,
            configName: this.options.configName,
            promptLinkingApp: false,
        });
        const postscript = outputContent `💡 To change these, run ${outputToken.packagejsonScript(this.app.packageManager, 'dev', '--reset')}`.value;
        let updateUrls;
        if (cachedInfo?.updateURLs === undefined) {
            updateUrls = NOT_CONFIGURED_TEXT;
        }
        else {
            updateUrls = cachedInfo.updateURLs ? 'Yes' : 'No';
        }
        let partnersAccountInfo = ['Partners account', 'unknown'];
        if (isServiceAccount(partnersSession.accountInfo)) {
            partnersAccountInfo = ['Service account', partnersSession.accountInfo.orgName];
        }
        else if (isUserAccount(partnersSession.accountInfo)) {
            partnersAccountInfo = ['Partners account', partnersSession.accountInfo.email];
        }
        const lines = [
            ['Configuration file', cachedInfo?.configFile || configurationFileNames.app],
            ['App name', cachedInfo?.title || NOT_CONFIGURED_TEXT],
            ['Client ID', cachedInfo?.appId || NOT_CONFIGURED_TEXT],
            ['Access scopes', getAppScopes(this.app.configuration)],
            ['Dev store', cachedInfo?.storeFqdn || NOT_CONFIGURED_TEXT],
            ['Update URLs', updateUrls],
            partnersAccountInfo,
        ];
        return [title, `${linesToColumns(lines)}\n\n${postscript}`];
    }
    projectSettingsSection() {
        const title = 'Your Project';
        const lines = [['Root location', this.app.directory]];
        return [title, linesToColumns(lines)];
    }
    async appComponentsSection() {
        const title = 'Directory Components';
        let body = this.webComponentsSection();
        function augmentWithExtensions(extensions, outputFormatter) {
            const types = new Set(extensions.map((ext) => ext.type));
            types.forEach((extensionType) => {
                const relevantExtensions = extensions.filter((extension) => extension.type === extensionType);
                if (relevantExtensions[0]) {
                    body += `\n\n${outputContent `${outputToken.subheading(relevantExtensions[0].externalType)}`.value}`;
                    relevantExtensions.forEach((extension) => {
                        body += `${outputFormatter(extension)}`;
                    });
                }
            });
        }
        const supportedExtensions = this.app.allExtensions.filter((ext) => ext.isReturnedAsInfo());
        augmentWithExtensions(supportedExtensions, this.extensionSubSection.bind(this));
        if (this.app.errors?.isEmpty() === false) {
            body += `\n\n${outputContent `${outputToken.subheading('Extensions with errors')}`.value}`;
            supportedExtensions.forEach((extension) => {
                body += `${this.invalidExtensionSubSection(extension)}`;
            });
        }
        return [title, body];
    }
    webComponentsSection() {
        const errors = [];
        const subtitle = [outputContent `${outputToken.subheading('web')}`.value];
        const toplevel = ['📂 web', ''];
        const sublevels = [];
        this.app.webs.forEach((web) => {
            if (web.configuration) {
                if (web.configuration.name) {
                    const { name, roles } = web.configuration;
                    sublevels.push([`    📂 ${name} (${roles.join(',')})`, relativePath(this.app.directory, web.directory)]);
                }
                else {
                    web.configuration.roles.forEach((role) => {
                        sublevels.push([`    📂 ${role}`, relativePath(this.app.directory, web.directory)]);
                    });
                }
            }
            else {
                sublevels.push([`  📂 ${UNKNOWN_TEXT}`, relativePath(this.app.directory, web.directory)]);
            }
            if (this.app.errors) {
                const error = this.app.errors.getError(`${web.directory}/${configurationFileNames.web}`);
                if (error)
                    errors.push(error);
            }
        });
        let errorContent = `\n${errors.map(this.formattedError).join('\n')}`;
        if (errorContent.trim() === '')
            errorContent = '';
        return `${subtitle}\n${linesToColumns([toplevel, ...sublevels])}${errorContent}`;
    }
    extensionSubSection(extension) {
        const config = extension.configuration;
        const details = [
            [`📂 ${extension.handle}`, relativePath(this.app.directory, extension.directory)],
            ['     config file', relativePath(extension.directory, extension.configurationPath)],
        ];
        if (config && config.metafields?.length) {
            details.push(['     metafields', `${config.metafields.length}`]);
        }
        return `\n${linesToColumns(details)}`;
    }
    invalidExtensionSubSection(extension) {
        const error = this.app.errors?.getError(extension.configurationPath);
        if (!error)
            return '';
        const details = [
            [`📂 ${extension.handle}`, relativePath(this.app.directory, extension.directory)],
            ['     config file', relativePath(extension.directory, extension.configurationPath)],
        ];
        const formattedError = this.formattedError(error);
        return `\n${linesToColumns(details)}\n${formattedError}`;
    }
    formattedError(str) {
        const [errorFirstLine, ...errorRemainingLines] = stringifyMessage(str).split('\n');
        const errorLines = [`! ${errorFirstLine}`, ...errorRemainingLines.map((line) => `  ${line}`)];
        return outputContent `${outputToken.errorText(errorLines.join('\n'))}`.value;
    }
    async systemInfoSection() {
        const title = 'Tooling and System';
        const { platform, arch } = platformAndArch();
        const versionUpgradeMessage = await this.versionUpgradeMessage();
        const cliVersionInfo = [this.currentCliVersion(), versionUpgradeMessage].join(' ').trim();
        const lines = [
            ['Shopify CLI', cliVersionInfo],
            ['Package manager', this.app.packageManager],
            ['OS', `${platform}-${arch}`],
            ['Shell', process.env.SHELL || 'unknown'],
            ['Node version', process.version],
        ];
        return [title, `${linesToColumns(lines)}`];
    }
    currentCliVersion() {
        return this.app.nodeDependencies['@shopify/cli'];
    }
    async versionUpgradeMessage() {
        const cliDependency = '@shopify/cli';
        const newestVersion = await checkForNewVersion(cliDependency, this.currentCliVersion());
        if (newestVersion) {
            return getOutputUpdateCLIReminder(this.app.packageManager, newestVersion);
        }
        return '';
    }
}
//# sourceMappingURL=info.js.map