import { getAvailableTCPPort } from '@shopify/cli-kit/node/tcp';
import { outputDebug, outputContent, outputToken, outputWarn } from '@shopify/cli-kit/node/output';
import * as http from 'http';
/**
 * A convenient function that runs an HTTP server and does path-based traffic forwarding to sub-processes that run
 * an HTTP server. The method assigns a random port to each of the processes.
 * @param tunnelUrl - The URL of the tunnel.
 * @param portNumber - The port to use for the proxy HTTP server. When undefined, a random port is automatically assigned.
 * @param proxyTargets - List of target processes to forward traffic to.
 * @param additionalProcesses - Additional processes to run. The proxy won't forward traffic to these processes.
 * @returns A promise that resolves with an interface to get the port of the proxy and stop it.
 */
export async function runConcurrentHTTPProcessesAndPathForwardTraffic({ portNumber, proxyTargets, additionalProcesses, abortController, }) {
    if (proxyTargets.length === 0) {
        return [...additionalProcesses];
    }
    const { rules, processDefinitions } = await createProcessDefinitionsForProxies(proxyTargets);
    outputDebug(outputContent `
Starting reverse HTTP proxy on port ${outputToken.raw(portNumber.toString())}
Routing traffic rules:
${outputToken.json(JSON.stringify(rules))}
`);
    const { server } = await getProxyingWebServer(rules, abortController.signal);
    return [
        {
            prefix: 'proxy',
            action: async () => {
                await server.listen(portNumber);
            },
        },
        ...processDefinitions,
        ...additionalProcesses,
    ];
}
export async function getProxyingWebServer(rules, abortSignal) {
    // Lazy-importing it because it's CJS and we don't want it
    // to block the loading of the ESM module graph.
    const { default: httpProxy } = await import('http-proxy');
    const proxy = httpProxy.createProxy();
    const server = http.createServer(getProxyServerRequestListener(rules, proxy));
    // Capture websocket requests and forward them to the proxy
    server.on('upgrade', getProxyServerWebsocketUpgradeListener(rules, proxy));
    abortSignal.addEventListener('abort', () => {
        outputDebug('Closing reverse HTTP proxy');
        server.close();
    });
    return { server };
}
function getProxyServerWebsocketUpgradeListener(rules, proxy) {
    return function (req, socket, head) {
        const target = match(rules, req, true);
        if (target) {
            return proxy.ws(req, socket, head, { target }, (err) => {
                outputWarn(`Error forwarding websocket request: ${err}`);
            });
        }
        socket.destroy();
    };
}
function getProxyServerRequestListener(rules, proxy) {
    return function (req, res) {
        const target = match(rules, req);
        if (target) {
            return proxy.web(req, res, { target }, (err) => {
                outputWarn(`Error forwarding web request: ${err}`);
            });
        }
        outputDebug(`
Reverse HTTP proxy error - Invalid path: ${req.url}
These are the allowed paths:
${outputToken.json(JSON.stringify(rules))}
`);
        res.statusCode = 500;
        res.end(`Invalid path ${req.url}`);
    };
}
async function createProcessDefinitionsForProxies(proxyTargets) {
    const rules = {};
    const createProxyProcessDefinition = async (target) => {
        const targetPort = target.customPort || (await getAvailableTCPPort());
        rules[target.pathPrefix ?? 'default'] = `http://localhost:${targetPort}`;
        const hmrServer = target.hmrServer;
        if (hmrServer) {
            rules.websocket = `http://localhost:${hmrServer.port}`;
            hmrServer.httpPaths.forEach((path) => (rules[path] = `http://localhost:${hmrServer.port}`));
        }
        return {
            prefix: target.logPrefix,
            action: async (stdout, stderr, signal) => {
                await target.action(stdout, stderr, signal, targetPort);
            },
        };
    };
    const proxyProcessDefinitions = proxyTargets.map(createProxyProcessDefinition);
    const processDefinitions = await Promise.all(proxyProcessDefinitions);
    return { rules, processDefinitions };
}
function match(rules, req, websocket = false) {
    const path = req.url ?? '/';
    for (const pathPrefix in rules) {
        if (path.startsWith(pathPrefix))
            return rules[pathPrefix];
    }
    if (websocket && rules.websocket)
        return rules.websocket;
    return rules.default;
}
//# sourceMappingURL=http-reverse-proxy.js.map