import { ConcurrentOutput } from './ConcurrentOutput.js';
import { render } from '../../testing/ui.js';
import { AbortController } from '../../../../public/node/abort.js';
import { unstyled } from '../../../../public/node/output.js';
import React from 'react';
import { describe, expect, test } from 'vitest';
describe('ConcurrentOutput', () => {
    test('renders a stream of concurrent outputs from sub-processes', async () => {
        // Given
        let backendPromiseResolve;
        let frontendPromiseResolve;
        const backendPromise = new Promise(function (resolve, _reject) {
            backendPromiseResolve = resolve;
        });
        const frontendPromise = new Promise(function (resolve, _reject) {
            frontendPromiseResolve = resolve;
        });
        const backendProcess = {
            prefix: 'backend',
            action: async (stdout, _stderr, _signal) => {
                stdout.write('first backend message');
                stdout.write('second backend message');
                stdout.write('third backend message');
                backendPromiseResolve();
            },
        };
        const frontendProcess = {
            prefix: 'frontend',
            action: async (stdout, _stderr, _signal) => {
                await backendPromise;
                stdout.write('first frontend message');
                stdout.write('second frontend message');
                stdout.write('third frontend message');
                frontendPromiseResolve();
                // await promise that never resolves
                await new Promise(() => { });
            },
        };
        // When
        const renderInstance = render(React.createElement(ConcurrentOutput, { processes: [backendProcess, frontendProcess], abortSignal: new AbortController().signal }));
        await frontendPromise;
        // Then
        expect(unstyled(renderInstance.lastFrame().replace(/\d/g, '0'))).toMatchInlineSnapshot(`
      "00:00:00 │ backend  │ first backend message
      00:00:00 │ backend  │ second backend message
      00:00:00 │ backend  │ third backend message
      00:00:00 │ frontend │ first frontend message
      00:00:00 │ frontend │ second frontend message
      00:00:00 │ frontend │ third frontend message
      "
    `);
    });
    test('rejects with the error thrown inside one of the processes', async () => {
        // Given
        const backendProcess = {
            prefix: 'backend',
            action: async (stdout, _stderr, _signal) => {
                stdout.write('first backend message');
                stdout.write('second backend message');
                stdout.write('third backend message');
                throw new Error('something went wrong');
            },
        };
        // When
        const renderInstance = render(React.createElement(ConcurrentOutput, { processes: [backendProcess], abortSignal: new AbortController().signal }));
        const renderPromise = renderInstance.waitUntilExit();
        await expect(renderPromise).rejects.toThrowError('something went wrong');
        expect(renderPromise.isRejected()).toBe(true);
    });
    test("doesn't reject when an error is thrown inside one of the processes and keepRunningAfterProcessesResolve is true", async () => {
        // Given
        const backendProcess = {
            prefix: 'backend',
            action: async (stdout, _stderr, _signal) => {
                stdout.write('first backend message');
                stdout.write('second backend message');
                stdout.write('third backend message');
                throw new Error('something went wrong');
            },
        };
        // When
        const renderInstance = render(React.createElement(ConcurrentOutput, { processes: [backendProcess], abortSignal: new AbortController().signal, keepRunningAfterProcessesResolve: true }));
        await new Promise((resolve) => setTimeout(resolve, 500));
        expect(renderInstance.waitUntilExit().isRejected()).toBe(false);
    });
    test('render promise resolves when all processes resolve by default', async () => {
        const backendProcess = {
            prefix: 'backend',
            action: async (stdout, _stderr, _signal) => {
                stdout.write('first backend message');
                stdout.write('second backend message');
                stdout.write('third backend message');
            },
        };
        // When
        const renderInstance = render(React.createElement(ConcurrentOutput, { processes: [backendProcess], abortSignal: new AbortController().signal }));
        const renderPromise = renderInstance.waitUntilExit();
        await renderPromise;
        expect(renderPromise.isFulfilled()).toBe(true);
    });
    test("render promise doesn't resolve when all processes resolve and keepRunningAfterProcessesResolve is true", async () => {
        const backendProcess = {
            prefix: 'backend',
            action: async (stdout, _stderr, _signal) => {
                stdout.write('first backend message');
                stdout.write('second backend message');
                stdout.write('third backend message');
            },
        };
        // When
        const renderInstance = render(React.createElement(ConcurrentOutput, { keepRunningAfterProcessesResolve: true, processes: [backendProcess], abortSignal: new AbortController().signal }));
        await new Promise((resolve) => setTimeout(resolve, 500));
        expect(renderInstance.waitUntilExit().isFulfilled()).toBe(false);
    });
});
//# sourceMappingURL=ConcurrentOutput.test.js.map