import type { MetricAttributes } from '@opentelemetry/api';
import type { MeterProvider, ViewOptions } from '@opentelemetry/sdk-metrics';
import type { MetricDescriptor, MetricRecording, MetricsConfig, OnRecordCallback, OtelService, RecordMetricFunction } from '../types.js';
export interface BaseOtelServiceOptions {
    /**
     * Service name is a unique name for an application/service.
     */
    serviceName: string;
    /**
     * If this is set to true then the service name is prefixed to every metric.
     */
    prefixMetric?: boolean;
    /**
     * Metrics to register on startup.
     */
    metrics?: MetricsConfig;
    /**
     * Called when a metric is recorded. `addOnRecord` can also be used to add
     * listeners anytime.
     */
    onRecord?: OnRecordCallback;
    /**
     * Override the default meter provider.
     */
    meterProvider?: MeterProvider;
}
export declare class BaseOtelService implements OtelService {
    readonly serviceName: string;
    readonly prefixMetric: boolean;
    protected readonly meterProvider: MeterProvider;
    protected readonly metrics: Map<string, RecordMetricFunction>;
    protected readonly recordListeners: Set<OnRecordCallback>;
    /**
     * Bootstraps an Otel exporter which can send Otel metrics to a dedicated Shopify supported collector endpoint.
     */
    constructor({ serviceName, prefixMetric, metrics, onRecord, meterProvider }: BaseOtelServiceOptions);
    getMeterProvider(): MeterProvider;
    addView(viewOptions: ViewOptions): void;
    record(metricName: string, value: number, labels?: MetricAttributes): void;
    registerMetric(metricName: string, { type, ...options }: MetricDescriptor): void;
    register(metrics: MetricsConfig): void;
    addOnRecord(onRecord: OnRecordCallback): () => void;
    removeOnRecord(onRecord: OnRecordCallback): void;
    shutdown(): Promise<void>;
    protected notifyRecordListeners(metricName: string, initialValue: number, initialLabels: MetricAttributes): MetricRecording;
}
