'use strict';

var React = require('react');
var polarisIcons = require('@shopify/polaris-icons');
var useToggle = require('../../../../utilities/use-toggle.js');
var breakpoints = require('../../../../utilities/breakpoints.js');
var css = require('../../../../utilities/css.js');
var FilterPill_module = require('./FilterPill.scss.js');
var hooks = require('../../../../utilities/i18n/hooks.js');
var Text = require('../../../Text/Text.js');
var InlineStack = require('../../../InlineStack/InlineStack.js');
var UnstyledButton = require('../../../UnstyledButton/UnstyledButton.js');
var Icon = require('../../../Icon/Icon.js');
var Button = require('../../../Button/Button.js');
var Popover = require('../../../Popover/Popover.js');
var BlockStack = require('../../../BlockStack/BlockStack.js');

function FilterPill({
  filterKey,
  label,
  filter,
  disabled,
  hideClearButton,
  selected,
  initialActive,
  closeOnChildOverlayClick,
  onRemove,
  onClick
}) {
  const i18n = hooks.useI18n();
  const {
    mdDown
  } = breakpoints.useBreakpoints();
  const elementRef = React.useRef(null);
  const {
    value: focused,
    setTrue: setFocusedTrue,
    setFalse: setFocusedFalse
  } = useToggle.useToggle(false);
  const [popoverActive, setPopoverActive] = React.useState(initialActive);
  React.useEffect(() => {
    const node = elementRef.current;
    if (!node || !popoverActive) {
      return;
    }
    const parent = node.parentElement?.parentElement;
    if (!parent) {
      return;
    }
    parent.scroll?.({
      left: node.offsetLeft
    });
  }, [elementRef, popoverActive]);
  const togglePopoverActive = React.useCallback(() => {
    if (filter) {
      setPopoverActive(popoverActive => !popoverActive);
    }
    if (onClick) {
      onClick(filterKey);
    }
  }, [filter, filterKey, onClick]);
  const handlePopoverClose = React.useCallback(() => {
    togglePopoverActive();
    if (!selected) {
      onRemove?.(filterKey);
    }
  }, [onRemove, selected, filterKey, togglePopoverActive]);
  const handleClear = () => {
    if (onRemove) onRemove(filterKey);
    setPopoverActive(false);
  };
  const buttonClasses = css.classNames(FilterPill_module.default.FilterButton, selected && FilterPill_module.default.ActiveFilterButton, popoverActive && FilterPill_module.default.FocusFilterButton, focused && FilterPill_module.default.focusedFilterButton);
  const clearButtonClassNames = css.classNames(FilterPill_module.default.PlainButton, FilterPill_module.default.clearButton);
  const toggleButtonClassNames = css.classNames(FilterPill_module.default.PlainButton, FilterPill_module.default.ToggleButton);
  const labelVariant = mdDown ? 'bodyLg' : 'bodySm';
  const wrappedLabel = /*#__PURE__*/React.createElement("div", {
    className: FilterPill_module.default.Label
  }, /*#__PURE__*/React.createElement(Text.Text, {
    variant: labelVariant,
    as: "span"
  }, label));
  const activator = /*#__PURE__*/React.createElement("div", {
    className: buttonClasses
  }, /*#__PURE__*/React.createElement(InlineStack.InlineStack, {
    gap: "0",
    wrap: false
  }, /*#__PURE__*/React.createElement(UnstyledButton.UnstyledButton, {
    onFocus: setFocusedTrue,
    onBlur: setFocusedFalse,
    onClick: togglePopoverActive,
    className: toggleButtonClassNames,
    type: "button"
  }, /*#__PURE__*/React.createElement(InlineStack.InlineStack, {
    wrap: false,
    align: "center",
    blockAlign: "center",
    gap: "0"
  }, selected ? /*#__PURE__*/React.createElement(React.Fragment, null, wrappedLabel) : /*#__PURE__*/React.createElement(React.Fragment, null, wrappedLabel, /*#__PURE__*/React.createElement("div", {
    className: FilterPill_module.default.IconWrapper
  }, /*#__PURE__*/React.createElement(Icon.Icon, {
    source: polarisIcons.ChevronDownIcon,
    tone: "base"
  }))))), selected ? /*#__PURE__*/React.createElement(UnstyledButton.UnstyledButton, {
    onClick: handleClear,
    className: clearButtonClassNames,
    type: "button",
    "aria-label": i18n.translate('Polaris.FilterPill.clear')
  }, /*#__PURE__*/React.createElement("div", {
    className: FilterPill_module.default.IconWrapper
  }, /*#__PURE__*/React.createElement(Icon.Icon, {
    source: polarisIcons.XSmallIcon,
    tone: "base"
  }))) : null));
  const clearButtonMarkup = !hideClearButton && /*#__PURE__*/React.createElement("div", {
    className: FilterPill_module.default.ClearButtonWrapper
  }, /*#__PURE__*/React.createElement(Button.Button, {
    onClick: handleClear,
    variant: "plain",
    disabled: !selected,
    textAlign: "left"
  }, i18n.translate('Polaris.FilterPill.clear')));
  if (disabled) {
    return null;
  }
  return /*#__PURE__*/React.createElement("div", {
    ref: elementRef
  }, /*#__PURE__*/React.createElement(Popover.Popover, {
    active: popoverActive,
    activator: activator,
    key: filterKey,
    onClose: handlePopoverClose,
    preferredAlignment: "left",
    preventCloseOnChildOverlayClick: !closeOnChildOverlayClick
  }, /*#__PURE__*/React.createElement("div", {
    className: FilterPill_module.default.PopoverWrapper
  }, /*#__PURE__*/React.createElement(Popover.Popover.Section, null, /*#__PURE__*/React.createElement(BlockStack.BlockStack, {
    gap: "100"
  }, filter, clearButtonMarkup)))));
}

exports.FilterPill = FilterPill;
