"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetSizeAppBlockJavaScript = void 0;
const types_1 = require("../../types");
const file_utils_1 = require("../../utils/file-utils");
const schema = {
    thresholdInBytes: types_1.SchemaProp.number(10000),
};
exports.AssetSizeAppBlockJavaScript = {
    meta: {
        code: 'AssetSizeAppBlockJavaScript',
        name: 'Asset Size App Block JavaScript',
        docs: {
            description: 'This check is aimed at preventing large JavaScript bundles from being included via Theme App Extensions.',
            url: 'https://shopify.dev/docs/themes/tools/theme-check/checks/asset-size-app-block-javascript',
            recommended: true,
        },
        type: types_1.SourceCodeType.LiquidHtml,
        severity: types_1.Severity.ERROR,
        schema,
        targets: [types_1.ConfigTarget.ThemeAppExtension],
    },
    create(context) {
        if (!context.fileSize) {
            return {};
        }
        return {
            async LiquidRawTag(node) {
                if (node.name !== 'schema')
                    return;
                let filePath;
                try {
                    filePath = JSON.parse(node.body.value).javascript;
                }
                catch (error) {
                    return;
                }
                if (!filePath) {
                    return;
                }
                const absolutePath = `assets/${filePath}`;
                const thresholdInBytes = context.settings.thresholdInBytes;
                const startIndex = node.body.position.start + node.body.value.indexOf(filePath);
                const endIndex = startIndex + filePath.length;
                const fileExists = await (0, file_utils_1.assertFileExists)(context, absolutePath);
                if (!fileExists) {
                    context.report({
                        message: `'${filePath}' does not exist.`,
                        startIndex: startIndex,
                        endIndex: endIndex,
                    });
                    return;
                }
                const fileSize = await context.fileSize(absolutePath);
                const fileExceedsThreshold = await (0, file_utils_1.assertFileSize)(thresholdInBytes, fileSize);
                if (fileExceedsThreshold) {
                    context.report({
                        message: `The file size for '${filePath}' (${fileSize} B) exceeds the configured threshold (${thresholdInBytes} B)`,
                        startIndex: startIndex,
                        endIndex: endIndex,
                    });
                }
            },
        };
    },
};
//# sourceMappingURL=index.js.map