"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LiquidHTMLSyntaxError = void 0;
const types_1 = require("../../types");
const utils_1 = require("../../utils");
function isParsingErrorWithLocation(error) {
    return 'name' in error && error.name === 'LiquidHTMLParsingError' && 'loc' in error;
}
function cleanErrorMessage(message, highlight) {
    return message
        .replace(/Line \d+, col \d+:\s+/, 'SyntaxError: ')
        .replace(/(?!<expected ".+",) not .*/, ` not "${highlight}"`);
}
exports.LiquidHTMLSyntaxError = {
    meta: {
        code: 'LiquidHTMLSyntaxError',
        aliases: ['SyntaxError', 'HtmlParsingError'],
        name: 'Prevent LiquidHTML Syntax Errors',
        docs: {
            description: 'This check exists to inform the user of Liquid HTML syntax errors.',
            recommended: true,
        },
        type: types_1.SourceCodeType.LiquidHtml,
        severity: types_1.Severity.ERROR,
        schema: {},
        targets: [],
    },
    create(context) {
        const error = context.file.ast;
        if (!(0, utils_1.isError)(error))
            return {};
        return {
            async onCodePathStart(file) {
                if (isParsingErrorWithLocation(error)) {
                    const { start, end } = error.loc;
                    const startIndex = (0, utils_1.getOffset)(file.source, start.line, start.column);
                    let endIndex = (0, utils_1.getOffset)(file.source, end.line, end.column);
                    if (startIndex === endIndex)
                        endIndex += 1;
                    const highlight = file.source.slice(startIndex, endIndex);
                    context.report({
                        message: cleanErrorMessage(error.message, highlight),
                        startIndex,
                        endIndex: endIndex,
                    });
                }
                else {
                    context.report({
                        message: error.message,
                        startIndex: 0,
                        endIndex: file.source.length,
                    });
                }
            },
        };
    },
};
//# sourceMappingURL=index.js.map