"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MatchingTranslations = void 0;
const types_1 = require("../../types");
const PLURALIZATION_KEYS = new Set(['zero', 'one', 'two', 'few', 'many', 'other']);
exports.MatchingTranslations = {
    meta: {
        code: 'MatchingTranslations',
        name: 'Translation files should have the same keys',
        docs: {
            description: 'TODO',
            recommended: true,
            url: 'https://shopify.dev/docs/themes/tools/theme-check/checks/matching-translations',
        },
        type: types_1.SourceCodeType.JSON,
        severity: types_1.Severity.ERROR,
        schema: {},
        targets: [],
    },
    create(context) {
        // State
        const defaultTranslations = new Set();
        const missingTranslations = new Set();
        const nodesByPath = new Map();
        // Helpers
        const hasDefaultTranslations = () => defaultTranslations.size > 0;
        const isTerminalNode = ({ type }) => type === 'Literal';
        const isPluralizationNode = (node) => PLURALIZATION_KEYS.has(node.key.value);
        const isShopifyPath = (path) => path.startsWith('shopify.');
        const hasDefaultTranslation = (translationPath) => { var _a; return (_a = defaultTranslations.has(translationPath)) !== null && _a !== void 0 ? _a : false; };
        const isDefaultTranslationsFile = ({ absolutePath }) => absolutePath.endsWith('.default.json');
        const isPluralizationPath = (path) => [...PLURALIZATION_KEYS].some((key) => path.endsWith(key));
        const isLocaleFile = ({ absolutePath }) => {
            const relativePath = context.relativePath(absolutePath);
            return relativePath.startsWith('locales/') && !relativePath.endsWith('schema.json');
        };
        const jsonPaths = (json) => {
            const keys = Object.keys(json);
            return keys.reduce((acc, key) => {
                if (typeof json[key] !== 'object') {
                    return acc.concat(key);
                }
                const childJson = json[key];
                const childPaths = jsonPaths(childJson);
                return acc.concat(childPaths.map((path) => `${key}.${path}`));
            }, []);
        };
        const objectPath = (nodes) => {
            return nodes
                .filter((node) => node.type === 'Property')
                .reduce((acc, val) => acc.concat(val.key.value), [])
                .join('.');
        };
        const countCommonParts = (arrayA, arrayB) => {
            const minLength = Math.min(arrayA.length, arrayB.length);
            for (let i = 0; i < minLength; i++) {
                if (arrayA[i] !== arrayB[i]) {
                    return i;
                }
            }
            return minLength;
        };
        const file = context.file;
        const ast = file.ast;
        if (!isLocaleFile(file) || isDefaultTranslationsFile(file) || ast instanceof Error) {
            // No need to lint a file that isn't a translation file, we return an
            // empty object as the check for those.
            return {};
        }
        const closestTranslationKey = (translationKey) => {
            var _a;
            const translationKeyParts = translationKey.split('.');
            let closestMatch = '';
            let maxCommonParts = 0;
            for (const path of nodesByPath.keys()) {
                const pathParts = path.split('.');
                const commonParts = countCommonParts(pathParts, translationKeyParts);
                if (commonParts > maxCommonParts) {
                    maxCommonParts = commonParts;
                    closestMatch = path;
                }
            }
            return (_a = nodesByPath.get(closestMatch)) !== null && _a !== void 0 ? _a : ast;
        };
        return {
            async onCodePathStart() {
                const defaultTranslationPaths = await context.getDefaultTranslations().then(jsonPaths);
                defaultTranslationPaths.forEach(Set.prototype.add, defaultTranslations);
                // At the `onCodePathStart`, we assume that all translations are missing,
                // and remove translation paths while traversing through the file.
                defaultTranslationPaths.forEach(Set.prototype.add, missingTranslations);
            },
            async Property(node, ancestors) {
                const path = objectPath(ancestors.concat(node));
                nodesByPath.set(path, node);
                if (!hasDefaultTranslations())
                    return;
                if (isPluralizationNode(node))
                    return;
                if (!isTerminalNode(node.value))
                    return;
                if (isShopifyPath(path))
                    return;
                if (hasDefaultTranslation(path)) {
                    // As `path` is present, we remove it from the
                    // `missingTranslationsPerFile` bucket.
                    missingTranslations.delete(path);
                    return;
                }
                context.report({
                    message: `A default translation for '${path}' does not exist`,
                    startIndex: node.loc.start.offset,
                    endIndex: node.loc.end.offset,
                    suggest: [
                        {
                            message: 'Delete unneeded translation key',
                            fix(corrector) {
                                corrector.remove(path);
                            },
                        },
                    ],
                });
            },
            async onCodePathEnd() {
                missingTranslations.forEach((path) => {
                    const closest = closestTranslationKey(path);
                    if (isPluralizationPath(path))
                        return;
                    if (isShopifyPath(path))
                        return;
                    context.report({
                        message: `The translation for '${path}' is missing`,
                        startIndex: closest.loc.start.offset,
                        endIndex: closest.loc.end.offset,
                        fix(corrector) {
                            corrector.add(path, 'TODO');
                        },
                    });
                });
            },
        };
    },
};
//# sourceMappingURL=index.js.map