"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidSchema = void 0;
const json_to_ast_1 = __importDefault(require("json-to-ast"));
const types_1 = require("../../types");
const json_schema_validate_utils_1 = require("../../utils/json-schema-validate-utils");
const find_node_at_path_1 = require("./find-node-at-path");
const parse_json_body_1 = require("./parse-json-body");
exports.ValidSchema = {
    meta: {
        code: 'ValidSchema',
        name: 'Prevent invalid JSON in {% schema %} tags',
        docs: {
            description: 'This check is aimed at eliminating JSON errors in schema tags.',
            recommended: true,
            url: 'https://shopify.dev/docs/themes/tools/theme-check/checks/valid-schema',
        },
        type: types_1.SourceCodeType.LiquidHtml,
        severity: types_1.Severity.ERROR,
        schema: {},
        targets: [],
    },
    create(context) {
        const filePath = context.file.absolutePath;
        const isSectionsDir = filePath.includes('/sections/');
        return {
            async LiquidRawTag(node) {
                var _a;
                if (node.name !== 'schema') {
                    return;
                }
                if (node.body.kind !== 'json') {
                    return;
                }
                const body = (0, parse_json_body_1.parseJsonBody)(node);
                if (body instanceof parse_json_body_1.JsonParseError) {
                    return context.report({
                        message: `Invalid syntax in schema JSON: ${body.message}`,
                        startIndex: body.position.start,
                        endIndex: body.position.end,
                    });
                }
                if (!isSectionsDir) {
                    /**
                     * Blocks for flex-sections, will use the schema tag but a different format than validateSectionSchema.
                     *
                     * TODO: We should support this case with its own bespoke JSON schema validation once
                     * blocks liquid schemas is finalized and documented.
                     */
                    return;
                }
                const validateSectionSchema = await ((_a = context.schemaValidators) === null || _a === void 0 ? void 0 : _a.validateSectionSchema());
                if (!validateSectionSchema) {
                    return;
                }
                const validate = (0, json_schema_validate_utils_1.withErrorFormatting)(validateSectionSchema);
                // Otherwise the schema is syntactically valid and we can perform further validation on the contents
                const errors = validate(body);
                const sectionSchemaAst = (0, json_to_ast_1.default)(node.body.value, { loc: true });
                // Character index of the start of the schema block. ie: where the ast parsing begins.
                const schemaStartIndex = node.blockStartPosition.end;
                errors.map(({ path, message }) => {
                    // Traverse the AST to find the node corresponding to the path
                    const pathSegments = path.split('.');
                    const node = (0, find_node_at_path_1.findNodeAtPath)(sectionSchemaAst, pathSegments);
                    if (!node) {
                        return;
                    }
                    let errorIndicies;
                    if ((0, types_1.isPropertyNode)(node)) {
                        const startLoc = node.key.loc;
                        const endLoc = node.value.loc;
                        errorIndicies = {
                            startIndex: schemaStartIndex + startLoc.start.offset,
                            endIndex: schemaStartIndex + endLoc.end.offset,
                        };
                    }
                    else {
                        const loc = node.loc;
                        errorIndicies = {
                            startIndex: schemaStartIndex + loc.start.offset,
                            endIndex: schemaStartIndex + loc.end.offset,
                        };
                    }
                    context.report({
                        message,
                        ...errorIndicies,
                    });
                });
            },
        };
    },
};
//# sourceMappingURL=index.js.map