"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.memoize = exports.memo = void 0;
const Unset = Symbol('Unset');
/** Returns a cached version of a function. Only caches one result. */
function memo(fn) {
    let cachedValue = Unset;
    const memoedFunction = (...args) => {
        if (cachedValue === Unset) {
            cachedValue = fn(...args);
        }
        return cachedValue;
    };
    memoedFunction.clearCache = () => {
        cachedValue = Unset;
    };
    return memoedFunction;
}
exports.memo = memo;
/**
 * Returns a function that is cached-by-keyFn(argument)
 *
 * e.g.
 *
 * const expensiveFunction = (thing: Thing) => ...
 * const thingToString = (thing: Thing): string => ...
 * const fastOnSubsequentCalls = memoize(
 *   expensiveFunction,
 *   thingToString,
 * );
 *
 * // slow first run
 * fastOnSubsequentCalls(thing1);
 *
 * // fast subsequent ones
 * fastOnSubsequentCalls(thing1);
 * fastOnSubsequentCalls(thing1);
 */
function memoize(fn, keyFn) {
    const cache = {};
    return (arg) => {
        const key = keyFn(arg);
        if (!cache[key]) {
            cache[key] = fn(arg);
        }
        return cache[key];
    };
}
exports.memoize = memoize;
//# sourceMappingURL=memo.js.map