"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThemeLiquidDocsManager = void 0;
const promises_1 = __importDefault(require("node:fs/promises"));
const jsonSchemaCompiler_1 = require("./jsonSchemaCompiler");
const themeLiquidDocsDownloader_1 = require("./themeLiquidDocsDownloader");
const utils_1 = require("./utils");
class ThemeLiquidDocsManager {
    constructor(log = utils_1.noop) {
        this.log = log;
        // These methods are memoized so that they both are lazy and cached with
        // minimal amount of state lying around.
        this.filters = (0, utils_1.memo)(async () => {
            return this.loadResource('filters', []);
        });
        this.objects = (0, utils_1.memo)(async () => {
            return this.loadResource('objects', []);
        });
        this.tags = (0, utils_1.memo)(async () => {
            return this.loadResource('tags', []);
        });
        this.systemTranslations = (0, utils_1.memo)(async () => {
            return this.loadResource('shopify_system_translations', {});
        });
        this.validateSectionSchema = (0, utils_1.memo)(async () => {
            const sectionSchema = await this.loadResource('section_schema', {});
            return (0, jsonSchemaCompiler_1.compileJsonSchema)(sectionSchema);
        });
        /**
         * The setup method checks that the latest revision matches the one from
         * Shopify/theme-liquid-docs. If there's a diff in revision, it means
         * that the documentations that you have locally are out of date.
         *
         * The setup method then downloads the other files.
         */
        this.setup = (0, utils_1.memo)(async () => {
            if (!(await (0, themeLiquidDocsDownloader_1.exists)(utils_1.root))) {
                await promises_1.default.mkdir(utils_1.root, { recursive: true });
            }
            const local = await this.latestRevision();
            await (0, utils_1.download)('latest');
            const remote = await this.latestRevision();
            if (local !== remote) {
                await Promise.all(themeLiquidDocsDownloader_1.Resources.map((resource) => (0, utils_1.download)(resource)));
            }
        });
    }
    async loadResource(name, defaultValue) {
        // Always wait for setup first. Since it's memoized, this will always
        // be the same promise.
        await this.setup();
        return this.load(name, defaultValue);
    }
    async load(name, defaultValue) {
        try {
            const content = await promises_1.default.readFile((0, utils_1.filePath)(name), 'utf8');
            const json = JSON.parse(content);
            return json;
        }
        catch (err) {
            this.log(`[SERVER] Error loading theme resource (${name}), ${err.message}`);
            return defaultValue;
        }
    }
    async latestRevision() {
        var _a;
        const latest = await this.load('latest', {});
        return (_a = latest['revision']) !== null && _a !== void 0 ? _a : '';
    }
}
exports.ThemeLiquidDocsManager = ThemeLiquidDocsManager;
//# sourceMappingURL=themeLiquidDocsManager.js.map