'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var react = require('react');
var errors = require('../errors.js');
var api = require('./api.js');
var subscription = require('./subscription.js');

/**
 * Returns the `buyerJourney` details on buyer progression in checkout.
 */
function useBuyerJourney() {
  const api$1 = api.useApi();
  if ('buyerJourney' in api$1) {
    return api$1.buyerJourney;
  }
  throw new errors.ExtensionHasNoMethodError('applyAttributeChange', api$1.extension.target);
}

/**
 * Returns true if the buyer completed submitting their order.
 *
 * For example, when viewing the **Order status** page after submitting payment, the buyer will have completed their order.
 */
function useBuyerJourneyCompleted() {
  const api$1 = api.useApi();
  if ('buyerJourney' in api$1) {
    return subscription.useSubscription(api$1.buyerJourney.completed);
  }
  throw new errors.ExtensionHasNoMethodError('buyerJourney', api$1.extension.target);
}

/**
 * Installs a function for intercepting and preventing progress on checkout.
 *
 * To block checkout progress, you must set the [block_progress](https://shopify.dev/docs/api/checkout-ui-extensions/configuration#block-progress)
 * capability in your extension's configuration.
 *
 * If you do, then you're expected to inform the buyer why navigation was blocked,
 * either by passing validation errors to the checkout UI or rendering the errors in your extension.
 */
function useBuyerJourneyIntercept(interceptor) {
  const api$1 = api.useApi();
  if (!('buyerJourney' in api$1)) {
    throw new errors.ExtensionHasNoMethodError('buyerJourney', api$1.extension.target);
  }
  const interceptorRef = react.useRef(interceptor);
  interceptorRef.current = interceptor;
  return react.useEffect(() => {
    const teardownPromise = api$1.buyerJourney.intercept(interceptorProps => interceptorRef.current(interceptorProps));
    return () => {
      teardownPromise.then(teardown => teardown()).catch(() => {});
    };
  }, [api$1.buyerJourney]);
}

/**
 * Returns all possible steps a buyer can take to complete the checkout. These steps may vary depending on the type of checkout or the shop's configuration.
 */
function useBuyerJourneySteps() {
  const api$1 = api.useApi();
  if (!('buyerJourney' in api$1) || !api$1.buyerJourney) {
    throw new errors.ExtensionHasNoMethodError('buyerJourney', api$1.extension.target);
  }
  return subscription.useSubscription(api$1.buyerJourney.steps);
}

/**
 * Returns the buyer journey step that the buyer is currently on.
 */
function useBuyerJourneyActiveStep() {
  const api$1 = api.useApi();
  if (!('buyerJourney' in api$1) || !api$1.buyerJourney) {
    throw new errors.ExtensionHasNoMethodError('buyerJourney', api$1.extension.target);
  }
  const steps = subscription.useSubscription(api$1.buyerJourney.steps);
  const activeStep = subscription.useSubscription(api$1.buyerJourney.activeStep);
  return activeStep ? steps.find(({
    handle
  }) => handle === activeStep.handle) : undefined;
}

exports.useBuyerJourney = useBuyerJourney;
exports.useBuyerJourneyActiveStep = useBuyerJourneyActiveStep;
exports.useBuyerJourneyCompleted = useBuyerJourneyCompleted;
exports.useBuyerJourneyIntercept = useBuyerJourneyIntercept;
exports.useBuyerJourneySteps = useBuyerJourneySteps;
