'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var errors = require('../errors.js');
var api = require('./api.js');
var subscription = require('./subscription.js');

/**
 * Returns the proposed `paymentAttributes` applied to the checkout.
 */
function usePaymentMethodAttributes() {
  const {
    paymentMethodAttributes
  } = api.useApi();
  if (!paymentMethodAttributes) {
    throw new errors.ScopeNotGrantedError('Using payment method attributes requires having checkout extension payments scope granted to your app.');
  }
  return subscription.useSubscription(paymentMethodAttributes);
}

/**
 * Returns the values for the specified `paymentAttributes` applied to the checkout.
 *
 * @param keys - An array of attribute keys.
 */
function usePaymentMethodAttributeValues(keys) {
  const attributes = usePaymentMethodAttributes();
  if (!(attributes !== null && attributes !== void 0 && attributes.length)) {
    return [];
  }
  return keys.map(key => {
    const attribute = attributes.find(attribute => attribute.key === key);
    return attribute === null || attribute === void 0 ? void 0 : attribute.value;
  });
}

/**
 * Returns a function to set payment method attributes.
 */
function useApplyPaymentMethodAttributesChange() {
  const api$1 = api.useApi();
  if (!api$1.applyPaymentMethodAttributesChange) {
    throw new errors.ExtensionHasNoMethodError('useApplyPaymentMethodAttributesChange', api$1.extension.target);
  }
  return api$1.applyPaymentMethodAttributesChange;
}

exports.useApplyPaymentMethodAttributesChange = useApplyPaymentMethodAttributesChange;
exports.usePaymentMethodAttributeValues = usePaymentMethodAttributeValues;
exports.usePaymentMethodAttributes = usePaymentMethodAttributes;
