'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var errors = require('../errors.js');
var api = require('./api.js');
var subscription = require('./subscription.js');

/**
 * Returns the current `Customer`.
 *
 * The value is `undefined` if the buyer isn't a known customer for this shop or if they haven't logged in yet.
 */
function useCustomer() {
  const buyerIdentity = api.useApi().buyerIdentity;
  if (!buyerIdentity) {
    throw new errors.ScopeNotGrantedError('Using buyer identity requires having personal customer data permissions granted to your app.');
  }
  return subscription.useSubscription(buyerIdentity.customer);
}

/**
 * Returns the email address of the buyer that is interacting with the cart.
 * The value is `undefined` if the app does not have access to customer data.
 */
function useEmail() {
  const buyerIdentity = api.useApi().buyerIdentity;
  if (!buyerIdentity) {
    throw new errors.ScopeNotGrantedError('Using buyer identity requires having personal customer data permissions granted to your app.');
  }
  return subscription.useSubscription(buyerIdentity.email);
}

/**
 * Returns the phone number of the buyer that is interacting with the cart.
 * The value is `undefined` if the app does not have access to customer data.
 */
function usePhone() {
  const buyerIdentity = api.useApi().buyerIdentity;
  if (!buyerIdentity) {
    throw new errors.ScopeNotGrantedError('Using buyer identity requires having personal customer data permissions granted to your app.');
  }
  return subscription.useSubscription(buyerIdentity.phone);
}

/**
 * Provides information about the company and its location that the business customer
 * is purchasing on behalf of during a B2B checkout. It includes details that can be utilized to
 * identify both the company and its corresponding location to which the business customer belongs.
 *
 * The value is `undefined` if a business customer isn't logged in. This function throws an error if the app doesn't have access to customer data.
 */
function usePurchasingCompany() {
  const buyerIdentity = api.useApi().buyerIdentity;
  if (!buyerIdentity) {
    throw new errors.ScopeNotGrantedError('Using buyer identity requires having personal customer data permissions granted to your app.');
  }
  return subscription.useSubscription(buyerIdentity.purchasingCompany);
}

exports.useCustomer = useCustomer;
exports.useEmail = useEmail;
exports.usePhone = usePhone;
exports.usePurchasingCompany = usePurchasingCompany;
