'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var api = require('./api.js');
var react = require('react');
var asyncSubscription = require('@remote-ui/async-subscription');

/**
 * Global instance of the subscribable that is created on the first `useStatefulSubscribableCart` call.
 * Use `destroyStatefulSubscribableCart` to destroy it and all of the subscribers.
 */
let statefulSubscribable;

/**
 * Verifies that the API has a Cart in it.
 */
const isCartApi = api => {
  return 'cart' in api;
};

/**
 * A hook utilizing `useState` and the `useStatefulSubscribableCart` function to create a component state.
 * @returns this hook returns the latest Cart state which re-renders on change.
 */
function useCartSubscription() {
  const statefulSubscribableCart = useStatefulSubscribableCart();
  const [cart, setCart] = react.useState(statefulSubscribableCart.current);
  const unsubscribeRef = react.useRef();
  react.useEffect(() => {
    if (!unsubscribeRef.current) {
      statefulSubscribableCart.subscribe(cart => {
        setCart(cart);
      });
    }
  }, [statefulSubscribableCart]);
  return cart;
}

/**
 * A hook utilizing the `makeStatefulSubscribable` function to allow multiple Cart subscriptions.
 * @returns StatefulRemoteSubscribable object with a Cart in it.
 */
function useStatefulSubscribableCart() {
  const api$1 = api.useApi();
  if (!isCartApi(api$1)) {
    throw new Error('No cart api found');
  }
  const {
    subscribable
  } = api$1.cart;
  if (!statefulSubscribable) {
    statefulSubscribable = asyncSubscription.makeStatefulSubscribable(subscribable);
  }
  return statefulSubscribable;
}
/**
 * A function destroying the subscriptions `useStatefulSubscribableCart` has.
 */
function destroyStatefulSubscribableCart() {
  var _statefulSubscribable;
  (_statefulSubscribable = statefulSubscribable) === null || _statefulSubscribable === void 0 ? void 0 : _statefulSubscribable.destroy();
  statefulSubscribable = undefined;
}

exports.destroyStatefulSubscribableCart = destroyStatefulSubscribableCart;
exports.useCartSubscription = useCartSubscription;
exports.useStatefulSubscribableCart = useStatefulSubscribableCart;
