'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var react = require('react');
var asyncSubscription = require('@remote-ui/async-subscription');
var api = require('./api.js');

/**
 * Global instance of the subscribable that is created on the first `useStatefulSubscribableLocale` call.
 * Use `destroyStatefulSubscribableLocale` to destroy it and all of the subscribers.
 */
let statefulSubscribable;

/**
 * Verifies that the API has a Locale in it.
 */
const isLocaleApi = api => {
  return 'locale' in api;
};

/**
 * A hook utilizing `useState` and the `useStatefulSubscribableLocale` function to create a component state.
 * @returns this hook returns the latest Locale state which re-renders on change.
 */
function useLocaleSubscription() {
  const statefulSubscribableLocale = useStatefulSubscribableLocale();
  const [locale, setLocale] = react.useState(statefulSubscribableLocale.current);
  const unsubscribeRef = react.useRef();
  react.useEffect(() => {
    if (!unsubscribeRef.current) {
      statefulSubscribableLocale.subscribe(locale => {
        setLocale(locale);
      });
    }
  }, [statefulSubscribableLocale]);
  return locale;
}

/**
 * A hook utilizing the `makeStatefulSubscribable` function to allow multiple Locale subscriptions.
 * @returns StatefulRemoteSubscribable object with a Locale in it.
 */
function useStatefulSubscribableLocale() {
  const api$1 = api.useApi();
  if (!isLocaleApi(api$1)) {
    throw new Error('No locale api found');
  }
  const {
    subscribable
  } = api$1.locale;
  if (!statefulSubscribable) {
    statefulSubscribable = asyncSubscription.makeStatefulSubscribable(subscribable);
  }
  return statefulSubscribable;
}
/**
 * A function destroying the subscriptions `useStatefulSubscribableCart` has.
 */
function destroyStatefulSubscribableLocale() {
  var _statefulSubscribable;
  (_statefulSubscribable = statefulSubscribable) === null || _statefulSubscribable === void 0 ? void 0 : _statefulSubscribable.destroy();
  statefulSubscribable = undefined;
}

exports.destroyStatefulSubscribableLocale = destroyStatefulSubscribableLocale;
exports.useLocaleSubscription = useLocaleSubscription;
exports.useStatefulSubscribableLocale = useStatefulSubscribableLocale;
