'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var react = require('react');
var asyncSubscription = require('@remote-ui/async-subscription');
var api = require('./api.js');

// Scanner Data Subscribable Hooks

/**
 * Global instance of the subscribable that is created on the first `useStatefulSubscribableScannerData` call.
 * Use `destroyStatefulSubscribableCart` to destroy it and all of the subscribers.
 */
let statefulScannerDataSubscribable;
const isScannerApi = api => {
  return 'scanner' in api;
};

/**
 * A hook utilizing `useState` and the `useStatefulSubscribableScannerData` function to create a component state.
 * @returns this hook returns the latest scan result state which re-renders on change.
 */
function useScannerDataSubscription() {
  const statefulSubscribableScanner = useStatefulSubscribableScannerData();
  const [scanResult, setScanResult] = react.useState(statefulSubscribableScanner.current);
  const unsubscribeRef = react.useRef();
  react.useEffect(() => {
    if (!unsubscribeRef.current) {
      statefulSubscribableScanner.subscribe(scanResult => {
        setScanResult(scanResult);
      });
    }
  }, [statefulSubscribableScanner]);
  return scanResult;
}

/**
 * A hook utilizing the `makeStatefulSubscribable` function to allow multiple scanner subscriptions.
 * @returns StatefulRemoteSubscribable object with a scan result in it.
 */
function useStatefulSubscribableScannerData() {
  const api$1 = api.useApi();
  if (!isScannerApi(api$1)) {
    throw new Error('No scanner api found');
  }
  const {
    scannerDataSubscribable
  } = api$1.scanner;
  if (!statefulScannerDataSubscribable) {
    statefulScannerDataSubscribable = asyncSubscription.makeStatefulSubscribable(scannerDataSubscribable);
  }
  return statefulScannerDataSubscribable;
}

/**
 * A function destroying the subscriptions `useStatefulSubscribableScannerData` has.
 */
function destroyStatefulSubscribableScannerData() {
  var _statefulScannerDataS;
  (_statefulScannerDataS = statefulScannerDataSubscribable) === null || _statefulScannerDataS === void 0 ? void 0 : _statefulScannerDataS.destroy();
  statefulScannerDataSubscribable = undefined;
}

// Scanner Sources Subscribable Hooks

/**
 * Global instance of the subscribable that is created on the first `useStatefulSubscribableScannerData` call.
 * Use `destroyStatefulSubscribableCart` to destroy it and all of the subscribers.
 */
let statefulScannerSourcesSubscribable;

/**
 * A hook utilizing `useState` and the `useStatefulSubscribableScannerData` function to create a component state.
 * @returns this hook returns the latest scan result state which re-renders on change.
 */
function useScannerSourcesSubscription() {
  const statefulSubscribableScannerSources = useStatefulSubscribableScannerSources();
  const [scannerSources, setScannerSources] = react.useState(statefulSubscribableScannerSources.current);
  const unsubscribeRef = react.useRef();
  react.useEffect(() => {
    if (!unsubscribeRef.current) {
      statefulSubscribableScannerSources.subscribe(scannerSources => {
        setScannerSources(scannerSources);
      });
    }
  }, [statefulSubscribableScannerSources]);
  return scannerSources;
}

/**
 * A hook utilizing the `makeStatefulSubscribable` function to allow multiple scanner subscriptions.
 * @returns StatefulRemoteSubscribable object with a scan result in it.
 */
function useStatefulSubscribableScannerSources() {
  const api$1 = api.useApi();
  if (!isScannerApi(api$1)) {
    throw new Error('No scanner api found');
  }
  const {
    scannerSourcesSubscribable
  } = api$1.scanner;
  if (!statefulScannerSourcesSubscribable) {
    statefulScannerSourcesSubscribable = asyncSubscription.makeStatefulSubscribable(scannerSourcesSubscribable);
  }
  return statefulScannerSourcesSubscribable;
}
/**
 * A function destroying the subscriptions `useStatefulSubscribableScannerData` has.
 */
function destroyStatefulSubscribableScannerSources() {
  var _statefulScannerSourc;
  (_statefulScannerSourc = statefulScannerSourcesSubscribable) === null || _statefulScannerSourc === void 0 ? void 0 : _statefulScannerSourc.destroy();
  statefulScannerSourcesSubscribable = undefined;
}

exports.destroyStatefulSubscribableScannerData = destroyStatefulSubscribableScannerData;
exports.destroyStatefulSubscribableScannerSources = destroyStatefulSubscribableScannerSources;
exports.useScannerDataSubscription = useScannerDataSubscription;
exports.useScannerSourcesSubscription = useScannerSourcesSubscription;
exports.useStatefulSubscribableScannerData = useStatefulSubscribableScannerData;
exports.useStatefulSubscribableScannerSources = useStatefulSubscribableScannerSources;
