import { useMemo } from 'react';
import { ExtensionHasNoMethodError } from '../errors.mjs';
import { useApi } from './api.mjs';
import { useSubscription } from './subscription.mjs';

/**
 * Returns all available payment options.
 */
function useAvailablePaymentOptions() {
  const api = useApi();
  if ('availablePaymentOptions' in api) {
    return useSubscription(api.availablePaymentOptions);
  }
  throw new ExtensionHasNoMethodError('availablePaymentOptions', api.extension.target);
}

/**
 * Returns payment options selected by the buyer.
 */
function useSelectedPaymentOptions() {
  const api = useApi();
  if (!('selectedPaymentOptions' in api) || !('availablePaymentOptions' in api)) {
    throw new ExtensionHasNoMethodError('selectedPaymentOptions', api.extension.target);
  }
  const selectedPaymentOptions = useSubscription(api.selectedPaymentOptions);
  const availablePaymentOptions = useSubscription(api.availablePaymentOptions);
  return useMemo(() => {
    const availablePaymentOptionsMap = {};
    for (const option of availablePaymentOptions) {
      availablePaymentOptionsMap[option.handle] = option;
    }
    return selectedPaymentOptions.map(paymentOption => {
      var _availablePaymentOpti;
      return {
        handle: paymentOption.handle,
        type: (_availablePaymentOpti = availablePaymentOptionsMap[paymentOption.handle]) === null || _availablePaymentOpti === void 0 ? void 0 : _availablePaymentOpti.type
      };
    });
  }, [availablePaymentOptions, selectedPaymentOptions]);
}

export { useAvailablePaymentOptions, useSelectedPaymentOptions };
