import { useApi } from './api.mjs';
import { useState, useRef, useEffect } from 'react';
import { makeStatefulSubscribable } from '@remote-ui/async-subscription';

/**
 * Global instance of the subscribable that is created on the first `useStatefulSubscribableCart` call.
 * Use `destroyStatefulSubscribableCart` to destroy it and all of the subscribers.
 */
let statefulSubscribable;

/**
 * Verifies that the API has a Cart in it.
 */
const isCartApi = api => {
  return 'cart' in api;
};

/**
 * A hook utilizing `useState` and the `useStatefulSubscribableCart` function to create a component state.
 * @returns this hook returns the latest Cart state which re-renders on change.
 */
function useCartSubscription() {
  const statefulSubscribableCart = useStatefulSubscribableCart();
  const [cart, setCart] = useState(statefulSubscribableCart.current);
  const unsubscribeRef = useRef();
  useEffect(() => {
    if (!unsubscribeRef.current) {
      statefulSubscribableCart.subscribe(cart => {
        setCart(cart);
      });
    }
  }, [statefulSubscribableCart]);
  return cart;
}

/**
 * A hook utilizing the `makeStatefulSubscribable` function to allow multiple Cart subscriptions.
 * @returns StatefulRemoteSubscribable object with a Cart in it.
 */
function useStatefulSubscribableCart() {
  const api = useApi();
  if (!isCartApi(api)) {
    throw new Error('No cart api found');
  }
  const {
    subscribable
  } = api.cart;
  if (!statefulSubscribable) {
    statefulSubscribable = makeStatefulSubscribable(subscribable);
  }
  return statefulSubscribable;
}
/**
 * A function destroying the subscriptions `useStatefulSubscribableCart` has.
 */
function destroyStatefulSubscribableCart() {
  var _statefulSubscribable;
  (_statefulSubscribable = statefulSubscribable) === null || _statefulSubscribable === void 0 ? void 0 : _statefulSubscribable.destroy();
  statefulSubscribable = undefined;
}

export { destroyStatefulSubscribableCart, useCartSubscription, useStatefulSubscribableCart };
