import { useState, useRef, useEffect } from 'react';
import { makeStatefulSubscribable } from '@remote-ui/async-subscription';
import { useApi } from './api.mjs';

/**
 * Global instance of the subscribable that is created on the first `useStatefulSubscribableConnectivity` call.
 * Use `destroyStatefulSubscribableConnectivity` to destroy it and all of the subscribers.
 */
let statefulSubscribable;

/**
 * Verifies that the API has a Connectivity in it.
 */
const isConnectivityApi = api => {
  return 'connectivity' in api;
};

/**
 * A hook utilizing `useState` and the `useStatefulSubscribableConnectivity` function to create a component state.
 * @returns this hook returns the latest Connectivity state which re-renders on change.
 */
function useConnectivitySubscription() {
  const statefulSubscribableConnectivity = useStatefulSubscribableConnectivity();
  const [connectivity, setConnectivity] = useState(statefulSubscribableConnectivity.current);
  const unsubscribeRef = useRef();
  useEffect(() => {
    if (!unsubscribeRef.current) {
      statefulSubscribableConnectivity.subscribe(connectivity => {
        setConnectivity(connectivity);
      });
    }
  }, [statefulSubscribableConnectivity]);
  return connectivity;
}

/**
 * A hook utilizing the `makeStatefulSubscribable` function to allow multiple Connectivity subscriptions.
 * @returns StatefulRemoteSubscribable object with a Connectivity in it.
 */
function useStatefulSubscribableConnectivity() {
  const api = useApi();
  if (!isConnectivityApi(api)) {
    throw new Error('No connectivity api found');
  }
  const {
    subscribable
  } = api.connectivity;
  if (!statefulSubscribable) {
    statefulSubscribable = makeStatefulSubscribable(subscribable);
  }
  return statefulSubscribable;
}

/**
 * A function destroying the subscriptions `useStatefulSubscribableConnectivity` has.
 */
function destroyStatefulSubscribableConnectivity() {
  var _statefulSubscribable;
  (_statefulSubscribable = statefulSubscribable) === null || _statefulSubscribable === void 0 ? void 0 : _statefulSubscribable.destroy();
  statefulSubscribable = undefined;
}

export { destroyStatefulSubscribableConnectivity, useConnectivitySubscription, useStatefulSubscribableConnectivity };
