import { useState, useRef, useEffect } from 'react';
import { makeStatefulSubscribable } from '@remote-ui/async-subscription';
import { useApi } from './api.mjs';

/**
 * Global instance of the subscribable that is created on the first `useStatefulSubscribableLocale` call.
 * Use `destroyStatefulSubscribableLocale` to destroy it and all of the subscribers.
 */
let statefulSubscribable;

/**
 * Verifies that the API has a Locale in it.
 */
const isLocaleApi = api => {
  return 'locale' in api;
};

/**
 * A hook utilizing `useState` and the `useStatefulSubscribableLocale` function to create a component state.
 * @returns this hook returns the latest Locale state which re-renders on change.
 */
function useLocaleSubscription() {
  const statefulSubscribableLocale = useStatefulSubscribableLocale();
  const [locale, setLocale] = useState(statefulSubscribableLocale.current);
  const unsubscribeRef = useRef();
  useEffect(() => {
    if (!unsubscribeRef.current) {
      statefulSubscribableLocale.subscribe(locale => {
        setLocale(locale);
      });
    }
  }, [statefulSubscribableLocale]);
  return locale;
}

/**
 * A hook utilizing the `makeStatefulSubscribable` function to allow multiple Locale subscriptions.
 * @returns StatefulRemoteSubscribable object with a Locale in it.
 */
function useStatefulSubscribableLocale() {
  const api = useApi();
  if (!isLocaleApi(api)) {
    throw new Error('No locale api found');
  }
  const {
    subscribable
  } = api.locale;
  if (!statefulSubscribable) {
    statefulSubscribable = makeStatefulSubscribable(subscribable);
  }
  return statefulSubscribable;
}
/**
 * A function destroying the subscriptions `useStatefulSubscribableCart` has.
 */
function destroyStatefulSubscribableLocale() {
  var _statefulSubscribable;
  (_statefulSubscribable = statefulSubscribable) === null || _statefulSubscribable === void 0 ? void 0 : _statefulSubscribable.destroy();
  statefulSubscribable = undefined;
}

export { destroyStatefulSubscribableLocale, useLocaleSubscription, useStatefulSubscribableLocale };
