<?php

namespace App\Http\Controllers;

use App\Jobs\TrackQTY;
use App\Models\Component;
use App\Models\Session;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class ComponentController extends Controller
{
    public function componentView(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'shop_name' => 'required'
        ]);
        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $response = [
                'status' => 'error',
                'message' => [$errors],
            ];
            return response()->json($response, 401);
        }
        try {
            $components = Component::where('shop_name', $request->shop_name)->orderBy('created_at','desc')->get()->toArray();

            if ($components) {
                $response = [
                    'status' => 'success',
                    'message' => 'Array of All Customers',
                    'data' => $components
                ];
                return response()->json($response, 200);
            } else {
                $response = [
                    'status' => 'error',
                    'message' => 'No Component Exists For This Shop',
                ];
                return response()->json($response, 404);
            }
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to Components', 'message' => $e->getMessage()], 500);
        }
    }

    #create component old api
    public function createComponent(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'shop_name' => 'required',
            'component_name' => 'required',
            'component_sku' => 'required|unique:components,component_sku',
            'component_stock_value' => 'numeric|required'
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $response = [
                'status' => 'error',
                'message' => $errors,
            ];
            return response()->json($response, 401);
        }

        try {
            $component = new Component;
            $component->shop_name = $request->shop_name;
            $component->component_name = $request->component_name;
            $component->component_sku = $request->component_sku;
            $component->component_stock_value = $request->component_stock_value;
            $component->save();

            $response = [
                'status' => 'success',
                'message' => 'Component Created Successfully',
                'shop_name' => $request->shop_name,
                'component_name' => $request->component_name,
                'component_sku' => $request->component_sku,
                'component_stock_value' => $request->component_stock_value
            ];

            return response()->json($response, 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to Create Components', 'message' => $e->getMessage()], 500);
        }
    }

    #create component new api
    public function createComponentNew(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'shop_name' => 'required',
            'component_name' => 'required',
            'component_sku' => 'required|unique:components,component_sku',
            'component_stock_value' => 'numeric|required',
            'three_month_stock' =>  'numeric|required'
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $response = [
                'status' => 'error',
                'message' => $errors,
            ];
            return response()->json($response, 401);
        }

        try {
            $component = new Component;
            $component->shop_name = $request->shop_name;
            $component->component_name = $request->component_name;
            $component->component_sku = $request->component_sku;
            $component->component_stock_value = $request->component_stock_value;
            $component->three_month_stock = $request->three_month_stock;
            $component->save();

            $response = [
                'status' => 'success',
                'message' => 'Component Created Successfully',
                'shop_name' => $request->shop_name,
                'component_name' => $request->component_name,
                'component_sku' => $request->component_sku,
                'component_stock_value' => $request->component_stock_value,
                'three_month_stock' => $request->three_month_stock,
            ];

            return response()->json($response, 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to Create Components', 'message' => $e->getMessage()], 500);
        }
    }

    #edit component old api
    public function editComponent(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'component_stock_value' => 'numeric',
            // 'component_sku' => 'unique:components,component_sku',
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $response = [
                'status' => 'error',
                'message' => [$errors],
            ];
            return response()->json($response, 401);
        }
        try {
            $componentId = Component::where('id', $request->id)->first();

            if (!$componentId) {
                $response = [
                    'status' => 'error',
                    'msg' => 'Id Not Found'
                ];
                return response()->json($response, 404);
            }
            $componentRequest = Component::find($request->id);

            if (!$componentRequest) {
                $response = [
                    'status' => 'error',
                    'msg' => 'No Component Found'
                ];
                return response()->json($response, 404);
            }

            $componentRequest->update([
                'component_name' => $request->component_name ?? $componentRequest->component_name,
                'component_sku' => $request->component_sku ?? $componentRequest->component_sku,
                'component_stock_value' => $request->component_stock_value ?? $componentRequest->component_stock_value,
            ]);

            return response()->json(['success' => 'true', 'message' => 'Component updated successfully'], 200);
        } catch (\Exception $e) {
            if ($e instanceof \Illuminate\Database\QueryException) {
                $errorCode = $e->errorInfo[1];
                if ($errorCode == 1062) {
                    $errorMessage = "Component SKU already exists.";
                } else {
                    $errorMessage = "Failed to Update Components";
                }
            } else {
                $errorMessage = "Failed to Update Components";
            }

            $response = [
                'status' => 'error',
                'message' => $errorMessage,
            ];
            return response()->json($response, 401);
        }

    }

    #edit component new api
    public function editComponentNew(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'component_stock_value' => 'numeric',
            'three_month_stock' => 'numeric'
            // 'component_sku' => 'unique:components,component_sku',
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $response = [
                'status' => 'error',
                'message' => $errors,
            ];
            return response()->json($response, 401);
        }
        try {
            $componentId = Component::where('id', $request->id)->first();

            if (!$componentId) {
                $response = [
                    'status' => 'error',
                    'msg' => 'Id Not Found'
                ];
                return response()->json($response, 404);
            }
            $componentRequest = Component::find($request->id);

            if (!$componentRequest) {
                $response = [
                    'status' => 'error',
                    'msg' => 'No Component Found'
                ];
                return response()->json($response, 404);
            }

            $componentRequest->update([
                'component_name' => $request->component_name ?? $componentRequest->component_name,
                'component_sku' => $request->component_sku ?? $componentRequest->component_sku,
                'component_stock_value' => $request->component_stock_value ?? $componentRequest->component_stock_value,
                'three_month_stock' => $request->three_month_stock ?? $componentRequest->three_month_stock,
            ]);

            return response()->json(['success' => 'true', 'message' => 'Component updated successfully'], 200);
        } catch (\Exception $e) {
            if ($e instanceof \Illuminate\Database\QueryException) {
                $errorCode = $e->errorInfo[1];
                if ($errorCode == 1062) {
                    $errorMessage = "Component SKU already exists.";
                } else {
                    $errorMessage = "Failed to Update Components";
                }
            } else {
                $errorMessage = "Failed to Update Components";
            }

            $response = [
                'status' => 'error',
                'message' => $errorMessage,
            ];
            return response()->json($response, 401);
        }

    }

    #delete component
    public function deleteComponent(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required',
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $response = [
                'status' => 'error',
                'message' => [$errors],
            ];
            return response()->json($response, 401);
        }

        try {
            $componentId = Component::where('id', $request->id)->delete();

            if (!$componentId) {
                $response = [
                    'status' => 'error',
                    'msg' => 'Something went wrong'
                ];
                return response()->json($response, 404);
            }

            return response()->json(['success' => 'true', 'message' => 'Component deleted successfully'], 200);
        } catch (\Exception $e) {
            $response = [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
            return response()->json($response, 401);
        }
    }

    public function getOrderId(Request $request){

        try{
            $orderId = $request->orderId;
            $shop = Session::pluck('shop')->first();

            $shopifyApiUrl = "https://{$shop}/admin/api/2023-10/orders/{$orderId}.json";

                $accessToken = Session::where('shop', $shop)->value('access_token');
                $response = Http::withHeaders([
                    'X-Shopify-Access-Token' => $accessToken,
                ])->get($shopifyApiUrl);

                $orderDetail = $response->json();
                $body = $orderDetail['order'];
                TrackQTY::dispatch($body,$shop);

            return response()->json(['status' => 'success','message' => 'Stock Updated Successfully'], 200);
        }
        catch(\Exception $e){
            return response()->json(['status' => 'error', 'message' => $e->getMessage()], 401);
        }
    }

}
