<?php

namespace App\Http\Controllers;

use App\Jobs\TrackOptionQTY;
use App\Models\Component;
use App\Models\ComponentMiniProduct;
use App\Models\MiniProduct;
use App\Models\OptionMiniProduct;
use App\Models\OptionProduct;
use App\Models\Options;
use App\Models\Products;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpKernel\Event\ResponseEvent;

class OptionsController extends Controller
{
    public function getOptions(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'shop_name' => 'required'
        ]);
        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $response = [
                'status' => 'error',
                'message' => [$errors],
            ];
            return response()->json($response, 401);
        }
        try {
            $options = Options::where('shop_name', $request->shop_name)->orderBy('created_at', 'desc')->get()->toArray();

            if ($options) {
                $response = [
                    'status' => 'success',
                    'message' => 'Array of All Options',
                    'data' => $options
                ];
                return response()->json($response, 200);
            } else {
                $response = [
                    'status' => 'success',
                    'message' => 'No Options Exists For This Shop',
                ];
                return response()->json($response, 200);
            }
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to Options', 'message' => [$e->getMessage()]], 500);
        }
    }

    public function createOptions(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'shop_name' => 'required',
                'option_name' => 'required|unique:options,option_name',
            ]);

            if ($validator->fails()) {
                $errors = $validator->errors()->all();
                $response = [
                    'status' => 'error',
                    'message' => [$errors],
                ];
                return response()->json($response, 401);
            }

            $addOption = new Options;
            $addOption->shop_name = $request->shop_name;
            $addOption->option_name = $request->option_name;
            $addOption->save();

            $response = [
                'status' => 'success',
                'message' => 'Option Created successfully',
            ];
            return response()->json($response, 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to Create Options', 'message' => [$e->getMessage()]], 500);
        }
    }

    public function createMiniProductComponentAndProduct(Request $request)
    {
        DB::beginTransaction();
        try {
            $validator = Validator::make($request->all(), [
                'shop_name' => 'required|string',
                'option_details' => 'required|array',
            ]);

            if ($validator->fails()) {
                return response()->json(['status' => 'error', 'message' => [$validator->errors()->all()]], 400);
            }

            $optionId = $request->option_details[0]['option_id'];

            $option = Options::where('id', $optionId)->exists();

            if ($option) {
                $miniProducts = $request->option_details[0]['values'];
                foreach ($miniProducts as $miniProduct) {
                    $addMiniProduct = new MiniProduct;
                    $addMiniProduct->mini_product = $miniProduct['name'];
                    $addMiniProduct->save();

                    $addOptionProduct = new OptionMiniProduct;
                    $addOptionProduct->option_id = $optionId;
                    $addOptionProduct->mini_product_id = $addMiniProduct->id;
                    $addOptionProduct->save();

                    $components = $miniProduct['components'];

                    foreach ($components as $component) {
                        $sku = $component['SKU'];
                        $quantity = $component['Quantity'];
                        $component = Component::where('component_sku', $sku)->first();

                        $checkComopnentExistsWithValue = ComponentMiniProduct::where('mini_product_id', $addMiniProduct->id)->where('component_id', $component->id)->exists();

                        if ($checkComopnentExistsWithValue) {
                            DB::rollBack();
                            return response()->json(['status' => 'error', 'message' => ['Component already exist with this value']], 500);
                        }
                        if ($component) {
                            $addComponentProduct = new ComponentMiniProduct;
                            $addComponentProduct->mini_product_id = $addMiniProduct->id;
                            $addComponentProduct->component_id = $component->id;
                            $addComponentProduct->quantity = $quantity;
                            $addComponentProduct->save();
                        } else {
                            DB::rollBack();
                            return response()->json(['status' => 'error', 'message' => ["Component does not exist for SKU: $sku"]], 500);
                        }
                    }
                }
                DB::commit();
                return response()->json(['status' => 'success', 'message' => ['Option, MiniProduct, and values added successfully']], 200);
            } else {
                return response()->json(['status' => 'error', 'message' => ['Option Id not found']], 400);
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['status' => 'error', 'message' => ['Failed to Assign Options Values and Mini Product'], 'error_details' => [$e->getMessage()]], 500);
        }
    }

    public function assignLookups(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'shop_name' => 'required|string',
                'option_id' => 'required',
                'product_ids' => 'required|array|min:1',
            ]);

            if ($validator->fails()) {
                return response()->json(['status' => 'error', 'message' => [$validator->errors()->all()]], 400);
            }

            $products = $request->product_ids;
            $optionId = $request->option_id;

            $checkOptionId = Options::where('id', $optionId)->exists();

            if ($checkOptionId) {
                foreach ($products as $product) {
                    $shop = Products::where('shop_name', $request->shop_name)->exists();
                    if ($shop) {
                        $productId = Products::where('product_id', $product)->exists();
                        $checkAlreadyAssigned = OptionProduct::where('product_id', $product)->where('option_id', $optionId)->exists();
                        if ($checkAlreadyAssigned) {
                            return response()->json(['status' => 'error', 'message' => ['Option Id ' . $optionId . ' already exists with the product Id ' . $productId]], 500);
                        }
                        if ($productId) {
                            $assignLookup = new OptionProduct;
                            $assignLookup->product_id = (int)$product;
                            $assignLookup->option_id = (int)$optionId;
                            $assignLookup->save();
                        } else {
                            $response = [
                                'status' => 'error',
                                'msg' => ['Product ID is missing or invalid'],
                            ];
                            return response()->json($response, 400);
                        }
                    } else {
                        return response()->json(['status' => 'error', 'message' => ['No Product Exist For this Shop']], 404);
                    }
                }
            } else {
                return response()->json(['status' => 'error', 'message' => ['Option ID Does not Exists or invalid']], 404);
            }

            $response = [
                'status' => 'success',
                'msg' => 'Lookups Assigned successfully',
            ];
            return response()->json($response, 200);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => [$e->getMessage()]], 500);
        }
    }

    public function deleteOption(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'shop_name' => 'required',
                'option_id' => 'required',
            ]);

            if ($validator->fails()) {
                return response()->json(['status' => 'error', 'message' => [$validator->errors()->all()]], 400);
            }

            $optionId = $request->option_id;
            $checkExists = Options::where('id', $optionId)->exists();

            if ($checkExists) {
                Options::where('id', $optionId)->delete();

                $response = [
                    'status' => 'success',
                    'msg' => 'Option Deleted successfully',
                ];
                return response()->json($response, 200);
            } else {
                return response()->json(['status' => 'error', 'message' => ['No Option Exist For this Shop']], 404);
            }
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => [$e->getMessage()]], 500);
        }
    }

    public function optionDetails(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'shop_name' => 'required',
                'option_id' => 'required',
            ]);

            if ($validator->fails()) {
                return response()->json(['status' => 'error', 'message' => [$validator->errors()->all()]], 400);
            }

            $checkShop = Options::where('shop_name', $request->shop_name)->exists();
            if ($checkShop) {
                $checkOptionId = Options::where('id', $request->option_id)->exists();
                if (!$checkOptionId) {
                    return response()->json(['status' => 'error', 'message' => ['Option Id not found']], 400);
                } else {
                    $optionDetail = Options::where('shop_name', $request->shop_name)
                        ->where('id', $request->option_id)
                        ->with(['miniProducts', 'components'])
                        ->first()
                        ->toArray();

                    $products = DB::table('products')
                        ->join('option_products', 'products.product_id', '=', 'option_products.product_id')
                        ->select('products.id', 'products.product_id', 'products.product_name')
                        ->where('option_products.option_id', '=', $request->option_id)
                        ->get()->toArray();
                    $optionName = Options::where('id', $request->option_id)->first();
                    $response = [
                        'status' => 'success',
                        'message' => 'Option Details',
                        'data' => [
                            'optionInfo' => [
                                [
                                    'option_id' => $request->option_id,
                                    'option_name' => $optionName->option_name
                                ]
                            ],
                            'mini_product' => $optionDetail['mini_products'],
                            'components' =>  $optionDetail['components'],
                            'lookups' => $products
                        ]
                    ];
                    return response()->json($response, 200);
                }
            } else {
                return response()->json(['status' => 'error', 'message' => ['Shop Does not Exists or Invalid']], 404);
            }
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => [$e->getMessage()]], 500);
        }
    }

    public function getProducts(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'shop_name' => 'required'
            ]);

            if ($validator->fails()) {
                return response()->json(['status' => 'error', 'message' => [$validator->errors()->all()]], 400);
            }

            $checkShop = Options::where('shop_name', $request->shop_name)->exists();

            if (!$checkShop) {
                return response()->json(['status' => 'error', 'message' => ['Shop Does not Exists or Invalid']], 404);
            } else {
                $products = Products::all()->toArray();
                if (!empty($products)) {
                    $response = [
                        'status' => 'success',
                        'message' => 'Product List',
                        'data' => $products
                    ];
                    return response()->json($response, 200);
                } else {
                    return response()->json(['status' => 'success', 'message' => ['No Products Exists']], 200);
                }
            }
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => [$e->getMessage()]], 500);
        }
    }
    public function removeAssignedProducts(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'shop_name' => 'required',
                'option_id' => 'required',
                'product_id' => 'required',
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => 'error', 'message' => [$validator->errors()->all()]], 400);
            }

            $checkShop = Options::where('shop_name', $request->shop_name)->exists();
            if (!$checkShop) {
                return response()->json(['status' => 'error', 'message' => ['Shop Does not Exists or Invalid']], 404);
            } else {
                $remove = OptionProduct::where('option_id', $request->option_id)
                    ->where('product_id', $request->product_id)
                    ->delete();
                if ($remove) {
                    return response()->json(['status' => 'success', 'message' => ['Product Removed Successfully']], 200);
                } else {
                    return response()->json(['status' => 'error', 'message' => ['Product Not Found']], 404);
                }
            }
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => [$e->getMessage()]], 500);
        }
    }

    public function editOption(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'shop_name' => 'required',
                'deletedComponents' => 'array',
                'deletedMiniProductIds' => 'array',
                'optionInfo' => 'array',
                'miniProducts' => 'array'
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => 'error', 'message' => [$validator->errors()->all()]], 400);
            }
            DB::beginTransaction();
            $checkShop = Options::where('shop_name', $request->shop_name)->exists();
            if (!$checkShop) {
                return response()->json(['status' => 'error', 'message' => ['Shop Does not Exists or Invalid']], 404);
            } else {
                //Update Option Name
                if (count($request->optionInfo) > 0) {
                    Options::where('id', $request->optionInfo[0]['option_id'])
                        ->update(['option_name' => $request->optionInfo[0]['option_name']]);
                }

                //Delete Components
                $deletedComponents = $request->deletedComponents;
                if (count($deletedComponents) > 0) {
                    foreach ($deletedComponents as $deletedComponent) {
                        $checkMiniProduct = MiniProduct::where('id', $deletedComponent['mini_product_id'])->exists();
                        if (!$checkMiniProduct) {
                            DB::rollBack();
                            return response()->json(['status' => 'error', 'message' => ['Mini Product Not Found for mini product id:' . $deletedComponent['mini_product_id']]], 404);
                        } else {
                            foreach ($deletedComponent['component_ids'] as $componentId) {
                                $checkComponent = ComponentMiniProduct::where('component_id', $componentId)->where('mini_product_id', $deletedComponent['mini_product_id'])->exists();
                                if (!$checkComponent) {
                                    DB::rollBack();
                                    return response()->json(['status' => 'error', 'message' => ['Component Id ' . $componentId . ' Not Found  for mini product Id:' . $deletedComponent['mini_product_id']]], 404);
                                } else {
                                    $remove = ComponentMiniProduct::where('mini_product_id', $deletedComponent['mini_product_id'])
                                        ->where('component_id', $componentId)
                                        ->delete();
                                    if ($remove === 0) {
                                        return response()->json(['status' => 'error', 'message' => 'Failed To Remove Components'], 500);
                                    }
                                }
                            }
                        }
                    }
                }
                //Delete Mini Products
                if (count($request->deletedMiniProductIds) > 0) {
                    foreach ($request->deletedMiniProductIds as $miniProductId) {
                        $checkExists = MiniProduct::where('id', $miniProductId)->exists();
                        if (!$checkExists) {
                            return response()->json(['status' => 'error', 'message' => 'Mini Product ID ' . $miniProductId . ' Not Found'], 404);
                        }
                        $remove =  MiniProduct::where('id', $miniProductId)->delete();

                        if ($remove === 0) {
                            return response()->json(['status' => 'error', 'message' => 'Failed To Remove Mini Products'], 500);
                        }
                    }
                }


                //Add Components
                foreach ($request->miniProducts as $miniProduct) {
                    if (!isset($miniProduct['id'])) {
                        $miniProductId = MiniProduct::create(['mini_product' => $miniProduct['mini_product']]);
                        foreach ($miniProduct['components'] as $component) {
                            $componentSKU = $component['component_sku'];
                            $quantity = $component['quantity'];
                            $check = Component::where('component_sku', $componentSKU)->exists();
                            $component = Component::where('component_sku', $componentSKU)->first();
                            if (!$check) {
                                DB::rollBack();
                                return response()->json(['status' => 'error', 'message' => ['Component Id ' . $component->id . ' Not Found']], 404);
                            }
                            ComponentMiniProduct::create(['component_id' => $component->id, 'mini_product_id' => $miniProductId->id, 'quantity' => $quantity]);
                        }
                        OptionMiniProduct::create(['option_id' => $request->optionInfo[0]['option_id'], 'mini_product_id' => $miniProductId->id]);
                    } else {
                        $miniProductId = $miniProduct['id'];
                        $checkComopnentExists = MiniProduct::where('id', $miniProductId)->exists();
                        if (!$checkComopnentExists) {
                            DB::rollBack();
                            return response()->json(['status' => 'error', 'message' => ['Mini Product Id not Exits For ID ' . $miniProductId]], 500);
                        }

                        //Update Mini Product
                        MiniProduct::where('id', $miniProductId)->update(['mini_product' => $miniProduct['mini_product']]);

                        foreach ($miniProduct['components'] as $component) {
                            $componentSKU = $component['component_sku'];
                            $quantity = $component['quantity'];
                            $check = Component::where('component_sku', $componentSKU)->exists();
                            if (!$check) {
                                DB::rollBack();
                                return response()->json(['status' => 'error', 'message' => ['Component Not Found']], 500);
                            }
                            $getComponent = Component::where('component_sku', $componentSKU)->first();
                            $componentId = $getComponent->id;
                            $checkComponent = ComponentMiniProduct::where('mini_product_id', $miniProductId)->where('component_id', $componentId)->exists();
                            if ($checkComponent) {
                                $update = ComponentMiniProduct::where('mini_product_id', $miniProductId)->where('component_id', $componentId)->update([
                                    'quantity' => $quantity
                                ]);
                            } else {
                                $add = new ComponentMiniProduct;
                                $add->mini_product_id = $miniProductId;
                                $add->component_id = $componentId;
                                $add->quantity = $quantity;
                                $add->save();
                            }
                        }
                    }
                }

                DB::commit();
                return response()->json(['status' => 'success', 'message' => 'Options Updated Successfully'], 200);
            }
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => [$e->getMessage()]], 500);
        }
    }
}
