import {
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  Text,
  IndexFiltersMode,
  useBreakpoints,
  Card,
  Layout,
  Frame,
  Page,
  Button,
  Box,
  Pagination,
  BlockStack,
  ButtonGroup,
} from "@shopify/polaris";
import { useState, useCallback, useEffect } from "react";
import EditComponent from "../components/EditComponent";
import { DeleteComponent } from "./DeleteComponent";
import { useFilteredData } from "../hooks";

export default function ComponentTable({ rows, Components }) {
  const [selected, setSelected] = useState(0);
  const [id, setId] = useState("");
  const [totalPages, setTotalPages] = useState();

  const [deleteComponentActive, setDeleteComponentActive] = useState(false);
  const [editComponentActive, setEditComponentActive] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);
  const PageSize = 50;
  const [value, setValue] = useState({
    sku: "",
    name: "",
    qty: "",
    stock: "",
  });

  const handleChange = (val, type) => {
    setValue({
      ...value,
      [type]: val,
    });
  };
  const handleNext = () => {
    setCurrentPage((prevPage) => prevPage + 1);
  };
  const handlePrevious = () => {
    setCurrentPage((prevPage) => prevPage - 1);
  };
  const sortOptions = [
    {
      label: "none",
      value: "none asc",
      directionLabel: "A-Z",
    },
    {
      label: "none",
      value: "none desc",
      directionLabel: "Z-A",
    },
    {
      label: "Component Name",
      value: "name asc",
      directionLabel: "A-Z",
    },
    {
      label: "Component Name",
      value: "name desc",
      directionLabel: "Z-A",
    },
    { label: "Component Sku", value: "sku asc", directionLabel: "A-Z" },
    { label: "Component Sku", value: "sku desc", directionLabel: "Z-A" },
    { label: "Component Stock", value: "stock asc", directionLabel: "A-Z" },
    {
      label: "Component Stock",
      value: "stock desc",
      directionLabel: "Z-A",
    },
  ];
  const [sortSelected, setSortSelected] = useState(["none"]);
  const { mode, setMode } = useSetIndexFiltersMode(IndexFiltersMode.Filtering);
  const onHandleCancel = () => {};

  const [taggedWith, setTaggedWith] = useState("");
  const [queryValue, setQueryValue] = useState("");

  const handleQueryValueChange = useCallback(
    (value) => {
      setQueryValue(value);

      let filteredRows = rows.filter(
        (item) =>
          item.component_name?.toLowerCase()?.includes(value.toLowerCase()) ||
          item.component_sku?.toLowerCase()?.includes(value.toLowerCase()) ||
          item.component_stock_value
            ?.toString()
            ?.toLowerCase()
            ?.includes(value.toLowerCase())
      );
      const totalPagesRounded = Math.ceil(filteredRows.length / PageSize);
      if (totalPagesRounded === 0) {
        setTotalPages(1);
        setCurrentPage(1);
      } else {
        setTotalPages(totalPagesRounded);
        setCurrentPage(1);
      }
    },
    [rows, PageSize, currentPage, totalPages]
  );
  const handleAccountStatusRemove = useCallback(() => setAccountStatus([]), []);
  const handleMoneySpentRemove = useCallback(
    () => setMoneySpent(undefined),
    []
  );
  const handleTaggedWithRemove = useCallback(() => setTaggedWith(""), []);
  const handleQueryValueRemove = useCallback(() => setQueryValue(""), []);
  const handleFiltersClearAll = useCallback(() => {
    handleAccountStatusRemove();
    handleMoneySpentRemove();
    handleTaggedWithRemove();
    handleQueryValueRemove();
  }, [
    handleQueryValueRemove,
    handleTaggedWithRemove,
    handleMoneySpentRemove,
    handleAccountStatusRemove,
  ]);

  const appliedFilters =
    taggedWith && !isEmpty(taggedWith)
      ? [
          {
            key: "taggedWith",
            label: disambiguateLabel("taggedWith", taggedWith),
            onRemove: handleTaggedWithRemove,
          },
        ]
      : [];

  const resourceName = {
    singular: "component",
    plural: "components",
  };
  const filteredData =
    rows.length > 0
      ? useFilteredData(rows, queryValue, sortSelected, currentPage, PageSize)
      : [];

  const handleEditComponent = (
    component_name,
    component_sku,
    component_stock_value,
    three_month_stock
  ) => {
    setEditComponentActive(true);
    setValue({
      name: component_name,
      sku: component_sku,
      qty: component_stock_value,
      stock: three_month_stock,
    });
  };
  const handleDeleteComponent = () => {
    setDeleteComponentActive(true);
  };
  useEffect(() => {
    const totalPagesRounded = Math.ceil(rows?.length / PageSize);
    if (totalPagesRounded === 0) {
      setTotalPages(1);
    } else {
      setTotalPages(Math.ceil(rows?.length / PageSize));
    }
  }, [PageSize, rows]);
  const rowMarkup = filteredData?.map((item, index) => (
    <IndexTable.Row id={item.id} key={item.id} position={index}>
      <IndexTable.Cell>
        <Text variant="bodyMd" fontWeight="bold" as="span">
          {item?.component_sku}
        </Text>
      </IndexTable.Cell>
      <IndexTable.Cell>{item?.component_name}</IndexTable.Cell>
      <IndexTable.Cell>{item?.component_stock_value}</IndexTable.Cell>
      <IndexTable.Cell>{item?.three_month_stock}</IndexTable.Cell>
      <IndexTable.Cell>
        <ButtonGroup>
          <Button
            variant="primary"
            onClick={() => {
              handleEditComponent(
                item.component_name,
                item.component_sku,
                item.component_stock_value,
                item.three_month_stock
              );
              setId(item.id);
            }}
          >
            Edit
          </Button>

          <Button
            variant="primary"
            onClick={() => {
              handleDeleteComponent();
              setId(item.id);
            }}
          >
            Delete
          </Button>
        </ButtonGroup>
      </IndexTable.Cell>
    </IndexTable.Row>
  ));

  return (
    <Frame>
      <Page fullWidth>
        <Layout>
          <Layout.Section>
            <Card padding={"0"}>
              {
                <IndexFilters
                  sortOptions={sortOptions}
                  sortSelected={sortSelected}
                  queryValue={queryValue}
                  queryPlaceholder="Searching in all"
                  onQueryChange={handleQueryValueChange}
                  onQueryClear={() => setQueryValue("")}
                  onSort={setSortSelected}
                  cancelAction={{
                    onAction: onHandleCancel,
                    disabled: false,
                    loading: false,
                  }}
                  tabs={[]}
                  selected={selected}
                  onSelect={setSelected}
                  canCreateNewView={false}
                  filters={[]}
                  appliedFilters={appliedFilters}
                  onClearAll={handleFiltersClearAll}
                  mode={mode}
                  setMode={setMode}
                />
              }
              <IndexTable
                condensed={useBreakpoints().smDown}
                resourceName={resourceName}
                itemCount={filteredData.length}
                selectable={false}
                headings={[
                  { title: "Component SKU" },
                  { title: "Component Name" },
                  {
                    title: "Component Stock QTY",
                  },
                  {
                    title: "3 Month Stock Threshold",
                  },
                  { title: "Action" },
                ]}
              >
                {rowMarkup}
              </IndexTable>

              <Box padding={"300"}>
                <BlockStack inlineAlign="center">
                  {rows?.length > PageSize && (
                    <Pagination
                      hasPrevious={currentPage !== 1}
                      onPrevious={handlePrevious}
                      hasNext={totalPages > currentPage}
                      onNext={handleNext}
                    />
                  )}
                </BlockStack>
              </Box>
            </Card>
          </Layout.Section>
        </Layout>
      </Page>
      {editComponentActive && (
        <EditComponent
          editComponentActive={editComponentActive}
          setEditComponentActive={setEditComponentActive}
          value={value}
          handleChange={handleChange}
          Components={Components}
          id={id}
        />
      )}
      {deleteComponentActive && (
        <DeleteComponent
          deleteComponentActive={deleteComponentActive}
          setDeleteComponentActive={setDeleteComponentActive}
          Components={Components}
          id={id}
        />
      )}
    </Frame>
  );

  function disambiguateLabel(key, value) {
    switch (key) {
      case "moneySpent":
        return `Money spent is between $${value[0]} and $${value[1]}`;
      case "taggedWith":
        return `Tagged with ${value}`;
      case "accountStatus":
        return value.map((val) => `Customer ${val}`).join(", ");
      default:
        return value;
    }
  }

  function isEmpty(value) {
    if (Array.isArray(value)) {
      return value.length === 0;
    } else {
      return value === "" || value == null;
    }
  }
}
