import { Listbox, Combobox, Icon } from "@shopify/polaris";
import { SearchIcon } from "@shopify/polaris-icons";
import { useState, useCallback, useMemo, useEffect } from "react";
import { useParams } from "react-router-dom";

export default function SelectOptions({ createdOptions, onOptionsChange }) {
    const deselectedOptions = useMemo(() => createdOptions, [createdOptions]);
    const id = useParams().idx;
    const [selectedOption, setSelectedOption] = useState();
    const [inputValue, setInputValue] = useState("");
    const [options, setOptions] = useState(deselectedOptions);
    const [loading, setLoading] = useState(false);

    const escapeSpecialRegExCharacters = useCallback(
        (value) => value?.replace(/[.*+?^${}()|[\]\\]/g, "\\$&"),
        []
    );

    const updateText = useCallback(
        (value) => {
            setInputValue(value);
            if (!loading) {
                setLoading(true);
            }
            setTimeout(() => {
                if (value === "") {
                    setOptions(deselectedOptions);
                    setSelectedOption("");
                    setLoading(false);
                    return;
                }
                const filterRegex = new RegExp(
                    escapeSpecialRegExCharacters(value),
                    "i"
                );
                const resultOptions = options.filter((option) =>
                    option.label.match(filterRegex)
                );
                setOptions(resultOptions);
                setLoading(false);
            }, 100);
        },
        [deselectedOptions, loading, options, escapeSpecialRegExCharacters]
    );

    const updateSelection = useCallback(
        (selected) => {
            const matchedOption = options.find((option) => {
                return option.value === selected;
            });

            setSelectedOption(selected);
            onOptionsChange(selected);
            setInputValue((matchedOption && matchedOption.label) || "");
        },
        [options]
    );
    useEffect(() => {
        if (id !== undefined && createdOptions) {
            const option_name = createdOptions.find(
                (item) => item.value === Number(id)
            );

            setInputValue(option_name?.label);
        }
    }, [id, createdOptions]);

    useEffect(() => {
        setOptions(deselectedOptions);
    }, [deselectedOptions]);
    const optionsMarkup =
        options?.length > 0
            ? options?.map((option) => {
                  const { label, value } = option;

                  return (
                      <Listbox.Option
                          key={`${value}`}
                          value={value}
                          selected={selectedOption === value}
                          accessibilityLabel={label}
                      >
                          {label}
                      </Listbox.Option>
                  );
              })
            : null;

    const loadingMarkup = loading ? (
        <Listbox.Loading accessibilityLabel="Loading" />
    ) : null;

    const listboxMarkup =
        optionsMarkup || loadingMarkup ? (
            <Listbox onSelect={updateSelection}>
                {optionsMarkup && !loading ? optionsMarkup : null}
                {loadingMarkup}
            </Listbox>
        ) : null;

    return (
        <Combobox
            activator={
                <Combobox.TextField
                    prefix={<Icon source={SearchIcon} />}
                    onChange={updateText}
                    label="Search Options"
                    labelHidden
                    value={inputValue}
                    placeholder="Search Options"
                    autoComplete="off"
                />
            }
        >
            {listboxMarkup}
        </Combobox>
    );
}
